/**
 * PazarRadar - Trendyol Content Script
 *
 * Trendyol.com sayfalarina enjekte edilen icerik scripti.
 * Urun verisini cikarir, analiz yapar ve PazarRadar arayuzunu sayfaya ekler.
 *
 * Desteklenen sayfa turleri:
 *   - Arama sonuclari (trendyol.com/sr?q=... veya kategori sayfalari)
 *   - Urun detay (trendyol.com/brand/product-p-12345)
 *   - Magaza sayfasi (trendyol.com/magaza/...)
 *
 * Bagimliliklar (manifest.json sirasiyla yuklenir):
 *   - window.PazarRadar.CommissionDB
 *   - window.PazarRadar.ProfitCalculator
 *   - window.PazarRadar.SalesEstimator
 *   - window.PazarRadar.InjectUI
 *
 * @namespace TrendyolContentScript
 */
(function () {
  'use strict';

  // ─── Kutuphane Referanslari ───────────────────────────────
  const PR = window.PazarRadar;

  if (!PR) {
    warn('PazarRadar namespace bulunamadi. Eklenti yuklenemedi.');
    return;
  }

  // ─── Sabitler ─────────────────────────────────────────────

  /** Platform tanimi */
  const PLATFORM = 'trendyol';

  /** Production modunda konsol ciktisini devre disi birakir */
  const DEBUG = false;
  const LOG_PREFIX = '[PazarRadar]';
  function warn(...args) { if (DEBUG) console.warn(LOG_PREFIX, ...args); }
  function info(...args) { if (DEBUG) console.info(LOG_PREFIX, ...args); }

  /** Lisans durumu - chrome.storage'dan yuklenir */
  let userIsPro = false;
  let userTrialDaysLeft = 0;

  try {
    if (typeof chrome !== 'undefined' && chrome.storage && chrome.storage.local) {
      chrome.storage.local.get('license', (result) => {
        const license = result && result.license;
        if (license && license.plan && license.plan !== 'free') {
          if (!license.validUntil || Date.now() <= license.validUntil) {
            userIsPro = true;
          }
          // Trial gun hesabi
          if (license.plan === 'trial' && license.trialEndsAt) {
            var msLeft = license.trialEndsAt - Date.now();
            userTrialDaysLeft = Math.max(0, Math.ceil(msLeft / (24 * 60 * 60 * 1000)));
            if (userTrialDaysLeft > 0) { userIsPro = true; }
          }
        }
      });
    }
  } catch (_e) { /* noop */ }

  /** SPA navigasyon algılama icin debounce suresi (ms) */
  const DEBOUNCE_DELAY = 500;

  /** Enjeksiyon performans limiti (ms) */
  const INJECTION_TIMEOUT = 200;

  /** Enjekte edilen elemanlara verilen isaretleme attribute */
  const INJECTED_ATTR = 'data-pazar-radar';

  /** PazarRadar tarafindan eklenmis elemanlari tanimlamak icin CSS sinifi */
  const PR_CLASS = 'pr-injected';

  /** Sayfa turleri */
  const PAGE_TYPES = Object.freeze({
    SEARCH: 'search',
    PRODUCT: 'product',
    STORE: 'store',
    UNKNOWN: 'unknown'
  });

  // ─── Arama sonuc sayfasi DOM selectorleri ─────────────────

  const SELECTORS = Object.freeze({
    // Arama sonuclari
    productCard: '.p-card-wrppr',
    productCardLink: '.p-card-wrppr a[href*="-p-"]',
    productCardName: '.prdct-desc-cntnr-name, .product-desc-sub-text',
    productCardPrice: '.prc-box-dscntd, .prc-box-sllng',
    productCardOriginalPrice: '.prc-box-orgnl',
    productCardRating: '.rating-score',
    productCardReviewCount: '.ratingCount',
    productCardSeller: '.merchant-text',
    productCardFavorite: '.fvrt-count, .favorite-count',
    productCardImage: '.p-card-img, img.product-image',
    searchResultsContainer: '.srch-rslt-out, .search-result-wrapper, .product-listing-container',
    breadcrumb: '.breadcrumb-wrapper, .breadcrumb, .srch-cntnt-brdc, .product-detail-breadcrumb',

    // Urun detay sayfasi (yeni micro-frontend + eski Next.js fallback)
    productTitle: 'h1.product-title, h1.pr-new-br, .product-detail-name, h1[class*="pr-"]',
    productPrice: 'p.new-price, .price-container .discounted, .price .discounted, .prc-dsc, .product-price-container .prc-box-dscntd',
    productOriginalPrice: 'p.old-price, .price-container .original, .price .original, .prc-org, .prc-box-orgnl',
    productRating: '.reviews-summary-average-rating, .rating-line-count, .ratings .score',
    productReviewCount: 'a.reviews-summary-reviews-detail, .rvw-cnt-tx, .total-rating-count',
    productSeller: '.merchant-name, a[href*="/magaza/"]',
    productFavorite: 'p.social-proof-content, .favorite-count',
    productBreadcrumb: '.breadcrumb-wrapper, .product-detail-breadcrumb, .breadcrumb',
    productDetailContainer: '.productDetailWrapper, .product-detail-container, #product-detail-app',

    // Magaza sayfasi
    storeName: '.merchant-header-name, .store-name',
    storeRating: '.merchant-header-rating, .store-rating',
    storeProductCount: '.merchant-product-count, .store-product-count',
    storeProducts: '.p-card-wrppr'
  });

  // ─── Durum Yonetimi ──────────────────────────────────────

  /** Aktif sayfa tipi */
  let currentPageType = PAGE_TYPES.UNKNOWN;

  /** Debounce zamanlayicisi */
  let debounceTimer = null;

  /** MutationObserver referansi */
  let contentObserver = null;

  /** Son islenen URL (gereksiz tekrar islemleri onlemek icin) */
  let lastProcessedUrl = '';

  /** Islem devam ediyor bayragi */
  let isProcessing = false;

  // ─── Sayfa Tipi Algilama ──────────────────────────────────

  /**
   * Mevcut URL'ye gore sayfa tipini belirler.
   * @returns {string} PAGE_TYPES enum degerlerinden biri
   */
  function detectPageType() {
    const url = window.location.href;
    const pathname = window.location.pathname;

    // Arama sonuclari: /sr?q=..., /arama?q=... veya /kategori--c-12345 desenleri
    if (url.includes('/sr?') || url.includes('/arama?') || pathname.match(/--c-\d+/)) {
      return PAGE_TYPES.SEARCH;
    }

    // Urun detay: URL icinde -p-12345 deseni
    if (pathname.match(/-p-\d+/)) {
      return PAGE_TYPES.PRODUCT;
    }

    // Magaza: /magaza/ yolu
    if (pathname.includes('/magaza/') || pathname.includes('/magaza-')) {
      return PAGE_TYPES.STORE;
    }

    return PAGE_TYPES.UNKNOWN;
  }

  /**
   * URL'den urun ID'sini cikarir.
   * @param {string} url - Urun URL'si
   * @returns {string|null} Urun ID
   */
  function extractProductIdFromUrl(url) {
    const match = (url || '').match(/-p-(\d+)/);
    return match ? match[1] : null;
  }

  // ─── __NEXT_DATA__ Veri Cikartma ─────────────────────────

  /**
   * Sayfadaki __NEXT_DATA__ script etiketinden JSON verisini cikarir.
   * @returns {Object|null} Parse edilmis JSON nesnesi veya null
   */
  function getNextData() {
    try {
      const scriptEl = document.getElementById('__NEXT_DATA__');
      if (!scriptEl || !scriptEl.textContent) {
        return null;
      }
      return JSON.parse(scriptEl.textContent);
    } catch (error) {
      warn('__NEXT_DATA__ parse hatasi:', error.message);
      return null;
    }
  }

  /**
   * __NEXT_DATA__'dan urun detay verisini cikarir.
   * Farkli veri yapilarina karsi dayanikli arama yapar.
   * @returns {Object|null} Normalize edilmis urun verisi
   */
  function extractProductFromNextData() {
    const nextData = getNextData();
    if (!nextData) {
      return null;
    }

    try {
      // Birincil yol: props.pageProps.product
      const pageProps = nextData.props && nextData.props.pageProps;
      if (!pageProps) {
        return null;
      }

      // Trendyol farkli property isimleri kullanabilir
      const product = pageProps.product
        || pageProps.productDetail
        || pageProps.initialData
        || (pageProps.productDetailResponse && pageProps.productDetailResponse.product);

      if (!product) {
        return null;
      }

      // Fiyat cikarma - birden fazla yapi desteklenir
      const priceData = product.price || product.priceInfo || {};
      const sellingPrice = priceData.sellingPrice
        || priceData.discountedPrice
        || priceData.originalPrice
        || product.price;

      const originalPrice = priceData.originalPrice || priceData.listPrice || sellingPrice;
      const discountedPrice = priceData.discountedPrice || priceData.sellingPrice || sellingPrice;

      // Derecelendirme cikarma
      const ratingData = product.ratingScore || product.rating || {};
      const averageRating = ratingData.averageRating || ratingData.average || 0;
      const totalRatingCount = ratingData.totalRatingCount || ratingData.totalCount || 0;
      const totalCommentCount = ratingData.totalCommentCount || ratingData.commentCount || 0;

      // Kategori cikarma
      const categoryData = product.category || {};
      const categoryHierarchy = categoryData.hierarchy || categoryData.name || '';
      const categoryId = categoryData.id || null;
      const categoryName = categoryData.name || '';

      // Marka cikarma
      const brandData = product.brand || {};
      const brandName = brandData.name || product.brandName || '';

      return {
        productId: String(product.id || product.productId || extractProductIdFromUrl(window.location.href) || ''),
        title: product.name || product.title || '',
        brand: brandName,
        category: categoryName,
        categoryHierarchy: categoryHierarchy,
        categoryId: categoryId,
        price: typeof sellingPrice === 'number' ? sellingPrice : parseFloat(sellingPrice) || 0,
        originalPrice: typeof originalPrice === 'number' ? originalPrice : parseFloat(originalPrice) || 0,
        discountedPrice: typeof discountedPrice === 'number' ? discountedPrice : parseFloat(discountedPrice) || 0,
        rating: averageRating,
        reviewCount: totalCommentCount || totalRatingCount,
        ratingCount: totalRatingCount,
        favoriteCount: product.favoriteCount || product.favCount || 0,
        sellerName: product.merchantName || (product.merchant && product.merchant.name) || '',
        sellerId: product.merchantId || (product.merchant && product.merchant.id) || null,
        productUrl: window.location.href,
        imageUrl: product.images && product.images.length > 0 ? product.images[0] : '',
        source: 'next_data'
      };
    } catch (error) {
      warn('__NEXT_DATA__ urun verisi cikarma hatasi:', error.message);
      return null;
    }
  }

  /**
   * __NEXT_DATA__'dan arama sonuclari urun listesini cikarir.
   * @returns {Array<Object>|null} Urun verisi dizisi veya null
   */
  function extractSearchResultsFromNextData() {
    const nextData = getNextData();
    if (!nextData) {
      return null;
    }

    try {
      const pageProps = nextData.props && nextData.props.pageProps;
      if (!pageProps) {
        return null;
      }

      // Arama sonuc verisinin olasi konumlari
      const searchData = pageProps.searchResult
        || pageProps.searchData
        || pageProps.initialSearchResult
        || pageProps.listingPageContent;

      if (!searchData) {
        return null;
      }

      const products = searchData.products
        || searchData.items
        || searchData.content
        || [];

      if (!Array.isArray(products) || products.length === 0) {
        return null;
      }

      return products.map(function (item) {
        const priceData = item.price || {};
        const ratingData = item.ratingScore || {};

        return {
          productId: String(item.id || item.productId || ''),
          title: item.name || item.title || '',
          brand: (item.brand && item.brand.name) || item.brandName || '',
          category: '',
          price: priceData.sellingPrice || priceData.discountedPrice || item.price || 0,
          originalPrice: priceData.originalPrice || priceData.listPrice || 0,
          discountedPrice: priceData.discountedPrice || priceData.sellingPrice || 0,
          rating: ratingData.averageRating || 0,
          reviewCount: ratingData.totalCommentCount || ratingData.totalRatingCount || 0,
          favoriteCount: item.favoriteCount || 0,
          sellerName: item.merchantName || '',
          productUrl: item.url ? ('https://www.trendyol.com' + item.url) : '',
          imageUrl: (item.images && item.images.length > 0) ? item.images[0] : '',
          source: 'next_data'
        };
      });
    } catch (error) {
      warn('__NEXT_DATA__ arama sonuclari cikarma hatasi:', error.message);
      return null;
    }
  }

  // ─── DOM Veri Cikartma ────────────────────────────────────

  /**
   * DOM elemani icerisinden text degerini guvenli sekilde alir.
   * @param {Element} container - Aranacak ust eleman
   * @param {string} selectorStr - CSS selector (virgul ile ayrilmis birden fazla olabilir)
   * @returns {string} Bulunan text veya bos string
   */
  function safeTextContent(container, selectorStr) {
    if (!container) {
      return '';
    }
    try {
      const el = container.querySelector(selectorStr);
      return el ? (el.textContent || '').trim() : '';
    } catch (_error) {
      return '';
    }
  }

  /**
   * Turk para birimi formatindaki fiyat stringini sayiya cevirir.
   * Ornekler: "1.299,90 TL" -> 1299.90, "899,99" -> 899.99
   * @param {string} priceStr - Fiyat metni
   * @returns {number} Sayisal deger veya 0
   */
  function parseTurkishPrice(priceStr) {
    if (!priceStr || typeof priceStr !== 'string') {
      return 0;
    }
    // TL, $ gibi para birimi sembollerini kaldir
    const cleaned = priceStr
      .replace(/[^\d.,]/g, '')
      .trim();

    if (!cleaned) {
      return 0;
    }

    // Turk formati: 1.299,90 (nokta binlik, virgul ondalik)
    // Once binlik noktalarini kaldir, sonra virgulen noktaya cevir
    const normalized = cleaned
      .replace(/\./g, '')
      .replace(',', '.');

    const value = parseFloat(normalized);
    return Number.isFinite(value) ? value : 0;
  }

  /**
   * Metin icerisindeki sayiyi cikarir (yorum sayisi, favori sayisi vb.)
   * Ornekler: "(123)" -> 123, "4.5K" -> 4500, "1.2B" -> 1200
   * @param {string} str - Sayı iceren metin
   * @returns {number} Cikartilmis sayi veya 0
   */
  function extractNumber(str) {
    if (!str || typeof str !== 'string') {
      return 0;
    }

    const cleaned = str.replace(/[()]/g, '').trim();

    // "1,2M" gibi milyon kisaltmasini isle
    var mMatch = cleaned.match(/([\d.,]+)\s*M/);
    if (mMatch) {
      return Math.round(parseFloat(mMatch[1].replace(',', '.')) * 1000000);
    }

    // "4.5K", "12,4B" gibi bin kisaltmalarini isle (B = Bin, K = Kilo/Thousand)
    var kMatch = cleaned.match(/([\d.,]+)\s*[kK]/);
    if (kMatch) {
      return Math.round(parseFloat(kMatch[1].replace(',', '.')) * 1000);
    }

    var bMatch = cleaned.match(/([\d.,]+)\s*B/);
    if (bMatch) {
      return Math.round(parseFloat(bMatch[1].replace(',', '.')) * 1000);
    }

    // Standart sayi cikarma
    var numStr = cleaned.replace(/[^\d]/g, '');
    var value = parseInt(numStr, 10);
    return Number.isFinite(value) ? value : 0;
  }

  /**
   * Breadcrumb'dan kategori bilgisini cikarir.
   * @returns {Object} { name: string, hierarchy: string, key: string }
   */
  function extractCategoryFromBreadcrumb() {
    try {
      const breadcrumbEl = document.querySelector(SELECTORS.breadcrumb);
      if (!breadcrumbEl) {
        return { name: '', hierarchy: '', key: '' };
      }

      // Yeni Trendyol: ul.breadcrumb-list > li > a yapisi
      // Oncelikle li icindeki a'lari al (tekrari onlemek icin)
      var items = breadcrumbEl.querySelectorAll('li > a');
      if (items.length === 0) {
        items = breadcrumbEl.querySelectorAll('a');
      }
      if (items.length === 0) {
        items = breadcrumbEl.querySelectorAll('span, li');
      }
      const parts = [];
      const seen = new Set();

      items.forEach(function (item) {
        const text = (item.textContent || '').trim();
        if (text && text !== '>' && text !== '/' && text.toLowerCase() !== 'trendyol' && !seen.has(text)) {
          seen.add(text);
          parts.push(text);
        }
      });

      // Son eleman genellikle en spesifik kategoridir
      const categoryName = parts.length > 0 ? parts[parts.length - 1] : '';
      const hierarchy = parts.join(' > ');

      // Kategori anahtari olusturma: "Elektronik > Telefon" -> "elektronik_telefon"
      const key = parts
        .map(function (p) {
          return p
            .toLowerCase()
            .replace(/[^a-z0-9\u00e7\u011f\u0131\u00f6\u015f\u00fc]/g, '_')
            .replace(/_+/g, '_')
            .replace(/^_|_$/g, '');
        })
        .filter(Boolean)
        .join('_');

      return { name: categoryName, hierarchy: hierarchy, key: key };
    } catch (_error) {
      return { name: '', hierarchy: '', key: '' };
    }
  }

  /**
   * URL yolundan kategori tahmin etmeye calisir.
   * Ornek: /giyim--c-1234 -> "giyim"
   * @returns {string} Tahmin edilen kategori adi
   */
  function extractCategoryFromUrl() {
    try {
      const pathname = window.location.pathname;

      // /kategori-adi--c-12345 formatı
      const catMatch = pathname.match(/\/([a-z0-9-]+)--c-\d+/i);
      if (catMatch) {
        return catMatch[1].replace(/-/g, ' ').trim();
      }

      // /sr?q=... durumunda breadcrumb'a geri don
      return '';
    } catch (_error) {
      return '';
    }
  }

  /**
   * DOM uzerinden urun detay sayfasindaki veriyi cikarir.
   * __NEXT_DATA__ kullanilamadigi durumlarda fallback olarak kullanilir.
   * @returns {Object|null} Normalize edilmis urun verisi
   */
  function extractProductFromDOM() {
    try {
      var title = safeTextContent(document, SELECTORS.productTitle);
      if (!title) {
        return null;
      }

      var priceText = safeTextContent(document, SELECTORS.productPrice);
      var originalPriceText = safeTextContent(document, SELECTORS.productOriginalPrice);
      var ratingText = safeTextContent(document, SELECTORS.productRating);
      var reviewCountText = safeTextContent(document, SELECTORS.productReviewCount);
      var sellerText = safeTextContent(document, SELECTORS.productSeller);

      // Favori sayisi - yeni Trendyol yapisi: "Sevilen urun! 1,2M kisi favoriledi!"
      var favoriteCount = 0;
      var favoriteEl = document.querySelector('p.social-proof-content');
      if (favoriteEl) {
        // Tum social-proof-content p etiketlerini tara, favoriledi icereni bul
        var socialProofs = document.querySelectorAll('p.social-proof-content');
        for (var i = 0; i < socialProofs.length; i++) {
          var spText = socialProofs[i].textContent || '';
          if (spText.includes('favoriledi')) {
            favoriteCount = extractNumber(spText);
            break;
          }
        }
      }
      if (favoriteCount === 0) {
        // Eski yapı fallback
        var oldFavText = safeTextContent(document, '.favorite-count');
        favoriteCount = extractNumber(oldFavText);
      }

      var price = parseTurkishPrice(priceText);
      // Yeni micro-frontend fallback: .price-wrapper icinden saf fiyat ara
      if (!price) {
        var priceWrapperEl = document.querySelector('.price-wrapper, .price');
        if (priceWrapperEl) {
          var leafEls = priceWrapperEl.querySelectorAll('p, span, div');
          for (var pi = 0; pi < leafEls.length; pi++) {
            if (leafEls[pi].childElementCount === 0) {
              var pTxt = (leafEls[pi].textContent || '').trim();
              if (pTxt.match(/^[\d.]+(?:,\d+)?\s*TL$/)) {
                price = parseTurkishPrice(pTxt);
                if (price > 0) break;
              }
            }
          }
        }
      }
      var originalPrice = parseTurkishPrice(originalPriceText) || price;

      var category = extractCategoryFromBreadcrumb();

      return {
        productId: extractProductIdFromUrl(window.location.href) || '',
        title: title,
        brand: extractBrandFromTitle(title),
        category: category.name,
        categoryHierarchy: category.hierarchy,
        categoryKey: category.key,
        categoryId: null,
        price: price,
        originalPrice: originalPrice,
        discountedPrice: price,
        rating: parseFloat(ratingText) || 0,
        reviewCount: extractNumber(reviewCountText),
        ratingCount: extractNumber(reviewCountText),
        favoriteCount: favoriteCount,
        sellerName: sellerText,
        sellerId: null,
        productUrl: window.location.href,
        imageUrl: '',
        source: 'dom'
      };
    } catch (error) {
      warn('DOM urun verisi cikarma hatasi:', error.message);
      return null;
    }
  }

  /**
   * Urun basligindan marka adini tahmin etmeye calisir.
   * Trendyol basliklarinda marka genellikle ilk kelimedir.
   * @param {string} title - Urun basligi
   * @returns {string} Tahmini marka adi
   */
  function extractBrandFromTitle(title) {
    if (!title) {
      return '';
    }
    // Ilk bosluktan onceki kismi marka olarak al (Trendyol genelde boyle yapilandirir)
    const parts = title.split(/\s+/);
    return parts.length > 0 ? parts[0] : '';
  }

  /**
   * DOM uzerinden tek bir urun karti icinden veri cikarir (arama sayfasi).
   * @param {Element} cardEl - Urun karti DOM elemani
   * @returns {Object|null} Urun verisi
   */
  function extractProductFromCard(cardEl) {
    if (!cardEl) {
      return null;
    }

    try {
      const linkEl = cardEl.querySelector('a[href*="-p-"]');
      const productUrl = linkEl ? linkEl.getAttribute('href') : '';
      const fullUrl = productUrl
        ? (productUrl.startsWith('http') ? productUrl : 'https://www.trendyol.com' + productUrl)
        : '';

      const title = safeTextContent(cardEl, SELECTORS.productCardName)
        || (linkEl ? (linkEl.getAttribute('title') || '') : '');

      const priceText = safeTextContent(cardEl, SELECTORS.productCardPrice);
      const originalPriceText = safeTextContent(cardEl, SELECTORS.productCardOriginalPrice);
      const ratingText = safeTextContent(cardEl, SELECTORS.productCardRating);
      const reviewCountText = safeTextContent(cardEl, SELECTORS.productCardReviewCount);
      const sellerText = safeTextContent(cardEl, SELECTORS.productCardSeller);
      const favoriteText = safeTextContent(cardEl, SELECTORS.productCardFavorite);

      const price = parseTurkishPrice(priceText);
      const originalPrice = parseTurkishPrice(originalPriceText) || price;

      // Indirim yuzdesi hesapla
      let discountPercent = 0;
      if (originalPrice > 0 && price > 0 && originalPrice > price) {
        discountPercent = Math.round(((originalPrice - price) / originalPrice) * 100);
      }

      // Gorsel URL
      const imgEl = cardEl.querySelector(SELECTORS.productCardImage);
      const imageUrl = imgEl
        ? (imgEl.getAttribute('src') || imgEl.getAttribute('data-src') || '')
        : '';

      return {
        productId: extractProductIdFromUrl(fullUrl) || '',
        title: title,
        brand: extractBrandFromTitle(title),
        category: '',
        price: price,
        originalPrice: originalPrice,
        discountedPrice: price,
        discountPercent: discountPercent,
        rating: parseFloat(ratingText) || 0,
        reviewCount: extractNumber(reviewCountText),
        favoriteCount: extractNumber(favoriteText),
        sellerName: sellerText,
        productUrl: fullUrl,
        imageUrl: imageUrl,
        source: 'dom_card'
      };
    } catch (error) {
      warn('Urun karti veri cikarma hatasi:', error.message);
      return null;
    }
  }

  // ─── Analiz ve Hesaplama ──────────────────────────────────

  /**
   * Urun verisi icin kar hesaplamasi yapar.
   * @param {Object} productData - Normalize edilmis urun verisi
   * @returns {Object|null} ProfitCalculator sonucu
   */
  function calculateProfit(productData) {
    if (!PR.ProfitCalculator || !productData || !productData.price) {
      return null;
    }

    try {
      const categoryKey = productData.categoryKey || productData.category || '';
      const categoryName = productData.categoryHierarchy || productData.category || '';

      return PR.ProfitCalculator.calculate({
        platform: PLATFORM,
        categoryKey: categoryKey,
        categoryName: categoryName,
        salePrice: productData.price,
        costPrice: productData.costPrice || null,
        cargoWeight: productData.cargoWeight || 1,
        kdvRate: 20,
        sellerPaysShipping: true
      });
    } catch (error) {
      warn('Kar hesaplama hatasi:', error.message);
      return null;
    }
  }

  /**
   * Urun verisi icin satis tahmini yapar.
   * @param {Object} productData - Normalize edilmis urun verisi
   * @returns {Object|null} SalesEstimator sonucu
   */
  function estimateSales(productData) {
    if (!PR.SalesEstimator || typeof PR.SalesEstimator.estimate !== 'function') {
      return null;
    }

    try {
      return PR.SalesEstimator.estimate({
        platform: PLATFORM,
        productId: productData.productId,
        title: productData.title,
        categoryKey: productData.categoryKey || '',
        currentPrice: productData.price,
        averageRating: productData.rating,
        totalReviews: productData.reviewCount,
        favoriteCount: productData.favoriteCount,
        sellerName: productData.sellerName
      });
    } catch (error) {
      warn('Satis tahmini hatasi:', error.message);
      return null;
    }
  }

  /**
   * Urun verisini analiz ederek tum hesaplamalari birlestirir.
   * @param {Object} productData - Normalize edilmis urun verisi
   * @returns {Object} Birlestirilmis analiz sonucu
   */
  function analyzeProduct(productData) {
    if (!productData) {
      return null;
    }

    const profitResult = calculateProfit(productData);
    const salesEstimate = estimateSales(productData);

    return {
      product: productData,
      profit: profitResult,
      sales: salesEstimate,
      platform: PLATFORM,
      analyzedAt: Date.now()
    };
  }

  // ─── Background Iletisimi ────────────────────────────────

  /**
   * Urun verisini background service worker'a gonderir (depolama icin).
   * @param {Object} productData - Urun verisi
   */
  function sendProductDataToBackground(productData) {
    if (!productData || !chrome.runtime || !chrome.runtime.sendMessage) {
      return;
    }

    try {
      chrome.runtime.sendMessage({
        type: 'product_data',
        platform: PLATFORM,
        data: {
          productId: productData.productId,
          title: productData.title,
          brand: productData.brand,
          category: productData.category,
          price: productData.price,
          originalPrice: productData.originalPrice,
          rating: productData.rating,
          reviewCount: productData.reviewCount,
          favoriteCount: productData.favoriteCount,
          sellerName: productData.sellerName,
          sellerRating: productData.sellerRating || null,
          timestamp: Date.now()
        }
      }, function (_response) {
        // Yanit varsa islenebilir, hata durumunda sessizce devam et
        if (chrome.runtime.lastError) {
          // Baglanti kapanmis olabilir, kritik degil
        }
      });
    } catch (_error) {
      // Background ile iletisim saglanamadiysa sessizce devam et
    }
  }

  /**
   * Kullanim takibi icin background'a mesaj gonderir.
   * @param {string} usageType - Kullanim tipi (ornek: 'product_analysis', 'search_analysis')
   */
  function trackUsage(usageType) {
    if (!chrome.runtime || !chrome.runtime.sendMessage) {
      return;
    }

    try {
      chrome.runtime.sendMessage({
        type: 'track_usage',
        payload: { type: usageType },
        platform: PLATFORM,
        timestamp: Date.now()
      }, function () {
        if (chrome.runtime.lastError) {
          // Sessizce devam et
        }
      });
    } catch (_error) {
      // Sessizce devam et
    }
  }

  // ─── Urun Skoru Hesaplama ─────────────────────────────────

  /**
   * Coklu faktorlu urun skoru hesaplar (0-10 arasi).
   * Faktorler: Rating (%25), Yorum hacmi (%20), Talep (%25), Trend (%15), Kar marji (%15)
   * @param {Object} params
   * @param {number} params.rating - Urun puani (0-5)
   * @param {number} params.reviewCount - Toplam yorum sayisi
   * @param {number} params.favoriteCount - Favori sayisi
   * @param {string} params.demandLevel - Talep seviyesi
   * @param {string} params.trend - Trend yonu ('up','down','stable')
   * @param {number} params.estimatedMonthlySales - Tahmini aylik satis
   * @param {number} params.price - Satis fiyati
   * @param {number} params.originalPrice - Orijinal fiyat (indirim oncesi)
   * @param {number} params.commissionRate - Komisyon orani (%)
   * @returns {number} 0-10 arasi skor
   */
  function calculateProductScore(params) {
    var scores = {};

    // 1. Rating skoru (%25 agirlik) - 0-5 arasi rating'i 0-10'a cevir
    var rating = Math.min(Math.max(params.rating || 0, 0), 5);
    if (rating === 0) {
      scores.rating = 5; // Rating yoksa notr (cezalandirma)
    } else if (rating >= 4.5) {
      scores.rating = 10;
    } else if (rating >= 4.0) {
      scores.rating = 8.5;
    } else if (rating >= 3.5) {
      scores.rating = 7;
    } else if (rating >= 3.0) {
      scores.rating = 5.5;
    } else if (rating >= 2.0) {
      scores.rating = 3.5;
    } else {
      scores.rating = 2;
    }

    // 2. Yorum hacmi skoru (%20 agirlik) - logaritmik olcekleme
    var reviews = params.reviewCount || 0;
    if (reviews >= 1000) {
      scores.reviews = 10;
    } else if (reviews >= 500) {
      scores.reviews = 9;
    } else if (reviews >= 200) {
      scores.reviews = 8;
    } else if (reviews >= 100) {
      scores.reviews = 7;
    } else if (reviews >= 50) {
      scores.reviews = 6;
    } else if (reviews >= 20) {
      scores.reviews = 5;
    } else if (reviews >= 5) {
      scores.reviews = 3.5;
    } else {
      scores.reviews = 2;
    }

    // 3. Talep seviyesi skoru (%25 agirlik)
    var demandScores = {
      'very_high': 10,
      'high': 8,
      'medium': 6,
      'low': 4,
      'very_low': 2
    };
    scores.demand = demandScores[params.demandLevel] || 5;

    // 4. Trend skoru (%15 agirlik)
    var trendScores = {
      'up': 9,
      'stable': 6,
      'down': 3
    };
    scores.trend = trendScores[params.trend] || 6;

    // 5. Fiyat/Deger skoru (%15 agirlik) - komisyon ve indirim durumu
    var commissionRate = params.commissionRate || 0;
    if (commissionRate <= 5) {
      scores.value = 9;
    } else if (commissionRate <= 10) {
      scores.value = 7.5;
    } else if (commissionRate <= 15) {
      scores.value = 6;
    } else if (commissionRate <= 25) {
      scores.value = 4.5;
    } else {
      scores.value = 3;
    }
    // Indirimli urunlere bonus (+1 max)
    if (params.originalPrice > 0 && params.price > 0 && params.originalPrice > params.price) {
      var discountPct = ((params.originalPrice - params.price) / params.originalPrice) * 100;
      if (discountPct >= 10) {
        scores.value = Math.min(10, scores.value + 1);
      }
    }

    // Agirlikli ortalama hesapla
    var weightedScore = (
      scores.rating * 0.25 +
      scores.reviews * 0.20 +
      scores.demand * 0.25 +
      scores.trend * 0.15 +
      scores.value * 0.15
    );

    // 0-5 araliginda sinirla (platformlarla uyumlu), 1 ondalik basamak
    var finalScore = Math.max(0, Math.min(10, weightedScore)) / 2;
    return Math.round(finalScore * 10) / 10;
  }

  // ─── UI Enjeksiyon ───────────────────────────────────────

  /**
   * Daha once enjekte edilmis PazarRadar elemanlarini sayfadan temizler.
   * Yeni sayfa navigasyonunda veya guncelleme oncesinde cagrilir.
   */
  function cleanupInjectedElements() {
    try {
      const injectedElements = document.querySelectorAll('[' + INJECTED_ATTR + ']');
      injectedElements.forEach(function (el) {
        if (el.parentNode) {
          el.parentNode.removeChild(el);
        }
      });
    } catch (_error) {
      // Temizleme sirasinda hata olusursa sessizce devam et
    }
  }

  /**
   * Bir DOM elemanini PazarRadar tarafindan enjekte edilmis olarak isaretler.
   * @param {Element} el - Isaretlenecek eleman
   * @returns {Element} Isaretlenmis eleman
   */
  function markAsInjected(el) {
    if (el) {
      el.setAttribute(INJECTED_ATTR, 'true');
      el.classList.add(PR_CLASS);
    }
    return el;
  }

  /**
   * Ic analiz nesnesini createProductPanel'in beklediği duz formata donusturur.
   * @param {Object} analysis - {product, profit, sales, platform} yapisi
   * @returns {Object} createProductPanel icin duz veri nesnesi
   */
  function flattenAnalysisForPanel(analysis) {
    const product = analysis.product || {};
    const profit = analysis.profit || {};
    const sales = analysis.sales || {};

    // Platform karsilastirmasi hesapla
    let comparison = null;
    if (PR.ProfitCalculator && typeof PR.ProfitCalculator.compareAll === 'function' && product.price > 0) {
      try {
        comparison = PR.ProfitCalculator.compareAll({
          categoryKey: product.categoryKey || product.category || '',
          categoryName: product.categoryHierarchy || product.category || '',
          salePrice: product.price,
          costPrice: product.costPrice || null,
          cargoWeight: product.cargoWeight || 1,
          kdvRate: 20,
          sellerPaysShipping: true
        });
      } catch (_e) {
        comparison = null;
      }
    }

    // Coklu faktorlu urun skoru hesaplama (0-10)
    var productScore = calculateProductScore({
      rating: product.rating || 0,
      reviewCount: product.reviewCount || 0,
      favoriteCount: product.favoriteCount || 0,
      demandLevel: sales.demandLevel || 'very_low',
      trend: sales.trend || 'stable',
      estimatedMonthlySales: sales.estimatedMonthlySales || 0,
      price: product.price || 0,
      originalPrice: product.originalPrice || 0,
      commissionRate: profit.commissionRate || 0
    });

    return {
      productScore: productScore,
      estimatedMonthlySales: sales.estimatedMonthlySales || 0,
      estimatedMonthlyRevenue: sales.estimatedMonthlyRevenue || 0,
      reviewRate: product.reviewCount || 0,
      favoriteRate: product.favoriteCount || 0,
      trend: sales.trend || 'stable',
      salePrice: profit.salePrice || product.price || 0,
      commissionRate: profit.commissionRate || 0,
      commissionAmount: profit.commissionAmount || 0,
      commissionKDV: profit.commissionKDV || 0,
      platformFee: profit.platformFee || 0,
      platformFeeKDV: profit.platformFeeKDV || 0,
      cargoFee: profit.estimatedCargo || 0,
      netRevenue: profit.netRevenue || 0,
      comparison: comparison,
      priceHistory: null,
      isPro: userIsPro,
      trialDaysLeft: userTrialDaysLeft
    };
  }

  /**
   * Urun detay sayfasina analiz panelini enjekte eder.
   * @param {Object} analysis - Birlestirilmis analiz sonucu
   */
  function injectProductPanel(analysis) {
    if (!PR.InjectUI || typeof PR.InjectUI.createProductPanel !== 'function') {
      warn('InjectUI.createProductPanel mevcut degil.');
      return;
    }

    try {
      // Ic analiz yapisini panel'in beklediği duz formata cevir
      const panelData = flattenAnalysisForPanel(analysis);

      // Limit kontrolu - background'a sor, sonra paneli render et
      if (typeof chrome !== 'undefined' && chrome.runtime && chrome.runtime.sendMessage) {
        chrome.runtime.sendMessage({
          type: 'peek_usage',
          payload: { usageType: 'productAnalyses' }
        }, function (response) {
          if (chrome.runtime.lastError || !response) {
            renderProductPanel(panelData);
            return;
          }
          if (!response.allowed) {
            panelData.limitReached = true;
            panelData.limitInfo = response;
          }
          renderProductPanel(panelData);
        });
      } else {
        renderProductPanel(panelData);
      }
    } catch (error) {
      warn('Urun paneli enjekte hatasi:', error.message);
    }
  }

  /**
   * Panel DOM'a ekleme yardimcisi.
   * @param {Object} panelData - Panel verisi
   */
  function renderProductPanel(panelData) {
    try {
      const panelEl = PR.InjectUI.createProductPanel(panelData);
      if (!panelEl) {
        return;
      }

      markAsInjected(panelEl);

      // Paneli urun detay sayfasinin sagina yerlestir
      const targetContainer = document.querySelector(SELECTORS.productDetailContainer)
        || document.querySelector('.product-detail')
        || document.querySelector('#product-detail-app')
        || document.querySelector('main');

      if (targetContainer) {
        targetContainer.appendChild(panelEl);
      } else {
        document.body.appendChild(panelEl);
      }
    } catch (error) {
      warn('Urun paneli render hatasi:', error.message);
    }
  }

  /**
   * Arama sonuclari sayfasinda her urun kartinin altina ozet kart enjekte eder.
   * @param {Element} cardEl - Urun karti DOM elemani
   * @param {Object} analysis - Birlestirilmis analiz sonucu
   */
  function injectSearchResultCard(cardEl, analysis) {
    if (!PR.InjectUI || typeof PR.InjectUI.createSearchResultCard !== 'function') {
      return;
    }

    try {
      // Bu kart daha once islenmis mi kontrol et
      if (cardEl.getAttribute(INJECTED_ATTR)) {
        return;
      }

      const resultCard = PR.InjectUI.createSearchResultCard(analysis);
      if (!resultCard) {
        return;
      }

      markAsInjected(resultCard);
      cardEl.setAttribute(INJECTED_ATTR, 'processed');

      // Kartın hemen altına ekle
      if (cardEl.nextSibling) {
        cardEl.parentNode.insertBefore(resultCard, cardEl.nextSibling);
      } else {
        cardEl.parentNode.appendChild(resultCard);
      }
    } catch (error) {
      warn('Arama sonuc karti enjekte hatasi:', error.message);
    }
  }

  /**
   * Arama sayfasinin ust kismina toplam analiz barini enjekte eder.
   * @param {Array<Object>} analyses - Tum urun analizleri
   */
  function injectSearchAggregateBar(analyses) {
    if (!PR.InjectUI || typeof PR.InjectUI.createSearchAggregateBar !== 'function') {
      return;
    }

    try {
      // Mevcut aggregate bar varsa kaldir
      const existingBar = document.querySelector('[' + INJECTED_ATTR + '="aggregate-bar"]');
      if (existingBar && existingBar.parentNode) {
        existingBar.parentNode.removeChild(existingBar);
      }

      // Istatistikleri hesapla
      const stats = computeSearchStats(analyses);
      const barEl = PR.InjectUI.createSearchAggregateBar(stats);
      if (!barEl) {
        return;
      }

      barEl.setAttribute(INJECTED_ATTR, 'aggregate-bar');
      barEl.classList.add(PR_CLASS);

      // Arama sonuc konteynerinin ustune yerlestir
      const resultsContainer = document.querySelector(SELECTORS.searchResultsContainer)
        || document.querySelector('.search-result')
        || document.querySelector('main');

      if (resultsContainer) {
        resultsContainer.insertBefore(barEl, resultsContainer.firstChild);
      }
    } catch (error) {
      warn('Arama istatistik bari enjekte hatasi:', error.message);
    }
  }

  /**
   * Arama sonuclari icin toplam istatistikleri hesaplar.
   * @param {Array<Object>} analyses - Analiz sonuclari dizisi
   * @returns {Object} Istatistik ozeti
   */
  function computeSearchStats(analyses) {
    const validAnalyses = analyses.filter(function (a) { return a && a.product; });
    const productCount = validAnalyses.length;

    if (productCount === 0) {
      return {
        productCount: 0,
        avgPrice: 0,
        minPrice: 0,
        maxPrice: 0,
        avgRating: 0,
        avgCommission: 0,
        platform: PLATFORM
      };
    }

    let totalPrice = 0;
    let minPrice = Infinity;
    let maxPrice = 0;
    let totalRating = 0;
    let ratingCount = 0;
    let totalCommission = 0;
    let commissionCount = 0;

    validAnalyses.forEach(function (a) {
      const price = a.product.price || 0;
      totalPrice += price;

      if (price > 0 && price < minPrice) {
        minPrice = price;
      }
      if (price > maxPrice) {
        maxPrice = price;
      }

      if (a.product.rating > 0) {
        totalRating += a.product.rating;
        ratingCount++;
      }

      if (a.profit && a.profit.commissionRate > 0) {
        totalCommission += a.profit.commissionRate;
        commissionCount++;
      }
    });

    return {
      productCount: productCount,
      avgPrice: productCount > 0 ? Math.round(totalPrice / productCount) : 0,
      minPrice: minPrice === Infinity ? 0 : minPrice,
      maxPrice: maxPrice,
      avgRating: ratingCount > 0 ? Math.round((totalRating / ratingCount) * 10) / 10 : 0,
      avgCommission: commissionCount > 0 ? Math.round((totalCommission / commissionCount) * 10) / 10 : 0,
      platform: PLATFORM
    };
  }

  // ─── Sayfa Isleyicileri ──────────────────────────────────

  /**
   * Urun detay sayfasini isle.
   * Oncelik: __NEXT_DATA__ > DOM parse
   */
  function handleProductPage() {
    function tryExtract() {
      // Oncelik 1: __NEXT_DATA__'dan veri cikar
      let productData = extractProductFromNextData();

      // Oncelik 2: DOM'dan fallback
      if (!productData) {
        productData = extractProductFromDOM();
      }

      if (!productData || !productData.title || !productData.price) {
        return null;
      }

      // Kategori bilgisini zenginlestir (gerekirse breadcrumb'dan)
      if (!productData.category && !productData.categoryHierarchy) {
        const breadcrumbCategory = extractCategoryFromBreadcrumb();
        productData = Object.assign({}, productData, {
          category: breadcrumbCategory.name,
          categoryHierarchy: breadcrumbCategory.hierarchy,
          categoryKey: breadcrumbCategory.key
        });
      }

      return productData;
    }

    const productData = tryExtract();
    if (productData) {
      const analysis = analyzeProduct(productData);
      injectProductPanel(analysis);
      sendProductDataToBackground(productData);
      trackUsage('product_analysis');
      return;
    }

    // SPA render gecikmesi icin kademeli retry (500, 1500, 3000, 5000ms)
    const retryDelays = [200, 600, 1500, 3500];
    let retryIndex = 0;

    function retryExtract() {
      if (retryIndex >= retryDelays.length) {
        warn('Urun verisi ' + retryDelays.length + ' denemede alinamadi.');
        return;
      }
      const delay = retryDelays[retryIndex];
      retryIndex++;
      setTimeout(function () {
        const retryData = tryExtract();
        if (retryData) {
          info(' Urun verisi ' + retryIndex + '. denemede alindi');
          const analysis = analyzeProduct(retryData);
          injectProductPanel(analysis);
          sendProductDataToBackground(retryData);
          trackUsage('product_analysis');
        } else {
          retryExtract();
        }
      }, delay);
    }

    retryExtract();
  }

  /**
   * Arama sonuclari sayfasini isle.
   * Once __NEXT_DATA__'dan denenir, basarisizsa DOM parse eder.
   */
  function handleSearchPage() {
    // Kategori bilgisini al (breadcrumb veya URL'den)
    const breadcrumbCategory = extractCategoryFromBreadcrumb();
    const urlCategory = extractCategoryFromUrl();
    const categoryName = breadcrumbCategory.name || urlCategory;
    const categoryKey = breadcrumbCategory.key || '';

    // Oncelik 1: __NEXT_DATA__'dan urun listesini al
    const nextDataProducts = extractSearchResultsFromNextData();

    // DOM'daki urun kartlarini bul
    const cardElements = document.querySelectorAll(SELECTORS.productCard);
    const analyses = [];

    if (nextDataProducts && nextDataProducts.length > 0) {
      // __NEXT_DATA__ verisi mevcut - kartlarla eslestirerek isle
      nextDataProducts.forEach(function (product, index) {
        // Kategori bilgisini ekle
        const enrichedProduct = Object.assign({}, product, {
          category: product.category || categoryName,
          categoryKey: categoryKey
        });

        const analysis = analyzeProduct(enrichedProduct);
        analyses.push(analysis);

        // Eslesen DOM kartini bul ve UI enjekte et
        if (index < cardElements.length) {
          injectSearchResultCard(cardElements[index], analysis);
        }
      });
    } else {
      // Fallback: DOM'dan urun kartlarini parse et
      cardElements.forEach(function (cardEl) {
        // Daha once islenmis karti atla
        if (cardEl.getAttribute(INJECTED_ATTR) === 'processed') {
          return;
        }

        const productData = extractProductFromCard(cardEl);
        if (!productData || !productData.price) {
          return;
        }

        // Kategori bilgisini ekle
        const enrichedProduct = Object.assign({}, productData, {
          category: productData.category || categoryName,
          categoryKey: categoryKey
        });

        const analysis = analyzeProduct(enrichedProduct);
        analyses.push(analysis);

        injectSearchResultCard(cardEl, analysis);
      });
    }

    // Arama istatistik barini enjekte et
    if (analyses.length > 0) {
      injectSearchAggregateBar(analyses);
    }

    trackUsage('search_analysis');
  }

  /**
   * Magaza sayfasini isle.
   * Magazanin genel bilgilerini cikarir ve urun kartlarini analiz eder.
   */
  function handleStorePage() {
    try {
      // Magaza bilgilerini cikar
      const storeName = safeTextContent(document, SELECTORS.storeName);
      const storeRating = safeTextContent(document, SELECTORS.storeRating);
      const productCountText = safeTextContent(document, SELECTORS.storeProductCount);

      const storeInfo = {
        name: storeName,
        rating: parseFloat(storeRating) || 0,
        productCount: extractNumber(productCountText),
        platform: PLATFORM,
        url: window.location.href
      };

      // Magaza urunlerini isle (arama sayfasiyla ayni mantik)
      const cardElements = document.querySelectorAll(SELECTORS.storeProducts);
      const analyses = [];

      cardElements.forEach(function (cardEl) {
        if (cardEl.getAttribute(INJECTED_ATTR) === 'processed') {
          return;
        }

        const productData = extractProductFromCard(cardEl);
        if (!productData || !productData.price) {
          return;
        }

        const analysis = analyzeProduct(productData);
        analyses.push(analysis);

        injectSearchResultCard(cardEl, analysis);
      });

      // Magaza ozet paneli enjekte et
      if (PR.InjectUI && typeof PR.InjectUI.createStorePanel === 'function') {
        const existingStorePanel = document.querySelector('[' + INJECTED_ATTR + '="store-panel"]');
        if (existingStorePanel && existingStorePanel.parentNode) {
          existingStorePanel.parentNode.removeChild(existingStorePanel);
        }

        const storePanel = PR.InjectUI.createStorePanel({
          store: storeInfo,
          analyses: analyses,
          stats: computeSearchStats(analyses)
        });

        if (storePanel) {
          storePanel.setAttribute(INJECTED_ATTR, 'store-panel');
          storePanel.classList.add(PR_CLASS);

          const mainContainer = document.querySelector('.merchant-container')
            || document.querySelector('.store-page')
            || document.querySelector('main');

          if (mainContainer) {
            mainContainer.insertBefore(storePanel, mainContainer.firstChild);
          }
        }
      }

      // Aggregate bar'i da goster (magaza urunleri icin)
      if (analyses.length > 0) {
        injectSearchAggregateBar(analyses);
      }

      trackUsage('store_analysis');
    } catch (error) {
      warn(' Magaza sayfasi isleme hatasi:', error.message);
    }
  }

  // ─── Ana Islem Dongusu ───────────────────────────────────

  /**
   * Sayfa icerigini isle.
   * Sayfa tipini belirler ve uygun isleyiciyi cagirir.
   */
  function processPage() {
    if (isProcessing) {
      return;
    }

    const currentUrl = window.location.href;

    // Ayni URL icin tekrar isleme yapma
    if (currentUrl === lastProcessedUrl) {
      // Ancak yeni urun kartlari eklenmis olabilir (infinite scroll)
      if (currentPageType === PAGE_TYPES.SEARCH || currentPageType === PAGE_TYPES.STORE) {
        handleNewCards();
      }
      return;
    }

    isProcessing = true;
    lastProcessedUrl = currentUrl;

    try {
      // Onceki enjeksiyonlari temizle (yeni sayfa navigasyonunda)
      cleanupInjectedElements();

      // Sayfa tipini belirle
      currentPageType = detectPageType();

      switch (currentPageType) {
        case PAGE_TYPES.PRODUCT:
          handleProductPage();
          break;

        case PAGE_TYPES.SEARCH:
          handleSearchPage();
          break;

        case PAGE_TYPES.STORE:
          handleStorePage();
          break;

        default:
          // Bilinmeyen sayfa tipi - islem yapma
          break;
      }
    } catch (error) {
      warn(' Sayfa isleme hatasi:', error.message);
    } finally {
      isProcessing = false;
    }
  }

  /**
   * Infinite scroll ile sayfaya eklenen yeni kartlari isle.
   * Sadece henuz islenmemis kartlari hedefler.
   */
  function handleNewCards() {
    if (isProcessing) {
      return;
    }

    isProcessing = true;

    try {
      const breadcrumbCategory = extractCategoryFromBreadcrumb();
      const urlCategory = extractCategoryFromUrl();
      const categoryName = breadcrumbCategory.name || urlCategory;
      const categoryKey = breadcrumbCategory.key || '';

      const allCards = document.querySelectorAll(SELECTORS.productCard);
      const newAnalyses = [];

      allCards.forEach(function (cardEl) {
        // Zaten islenmis kartlari atla
        if (cardEl.getAttribute(INJECTED_ATTR) === 'processed') {
          return;
        }

        const productData = extractProductFromCard(cardEl);
        if (!productData || !productData.price) {
          return;
        }

        const enrichedProduct = Object.assign({}, productData, {
          category: productData.category || categoryName,
          categoryKey: categoryKey
        });

        const analysis = analyzeProduct(enrichedProduct);
        newAnalyses.push(analysis);

        injectSearchResultCard(cardEl, analysis);
      });

      // Yeni kartlar islendiyse aggregate bar'i guncelle
      if (newAnalyses.length > 0) {
        // Tum mevcut analizleri topla (mevcut + yeni)
        // Basitlik icin sadece aggregate bar'i yeniden hesapla
        handleSearchAggregateUpdate();
      }
    } catch (error) {
      warn(' Yeni kart isleme hatasi:', error.message);
    } finally {
      isProcessing = false;
    }
  }

  /**
   * Aggregate bar'i sayfadaki tum islenmis kartlara gore gunceller.
   */
  function handleSearchAggregateUpdate() {
    try {
      const allCards = document.querySelectorAll(SELECTORS.productCard + '[' + INJECTED_ATTR + '="processed"]');
      const allAnalyses = [];

      allCards.forEach(function (cardEl) {
        const productData = extractProductFromCard(cardEl);
        if (productData && productData.price) {
          allAnalyses.push(analyzeProduct(productData));
        }
      });

      if (allAnalyses.length > 0) {
        injectSearchAggregateBar(allAnalyses);
      }
    } catch (_error) {
      // Guncelleme hatasi kritik degil
    }
  }

  // ─── SPA Navigasyon Izleme ───────────────────────────────

  /**
   * Debounce ile sayfa islemesini tetikler.
   * Arka arkaya gelen mutation'lari birlestirerek gereksiz tekrari onler.
   */
  function debouncedProcessPage() {
    if (debounceTimer) {
      clearTimeout(debounceTimer);
    }

    debounceTimer = setTimeout(function () {
      debounceTimer = null;

      // Performans icin requestIdleCallback kullan (destekleniyorsa)
      if (typeof requestIdleCallback === 'function') {
        requestIdleCallback(function () {
          processPage();
        }, { timeout: INJECTION_TIMEOUT });
      } else {
        setTimeout(processPage, 0);
      }
    }, DEBOUNCE_DELAY);
  }

  /**
   * MutationObserver'i baslatir.
   * Ana icerik alanini izleyerek SPA navigasyonlarini yakalar.
   */
  function startContentObserver() {
    if (contentObserver) {
      contentObserver.disconnect();
    }

    // Gozlemlenecek hedef: ana icerik alani veya body
    const targetNode = document.querySelector('#app')
      || document.querySelector('#__next')
      || document.querySelector('main')
      || document.body;

    if (!targetNode) {
      warn(' Icerik gozlemleme hedefi bulunamadi.');
      return;
    }

    contentObserver = new MutationObserver(function (mutations) {
      // Sadece PazarRadar tarafindan yapilmamis degisiklikleri dikkate al
      let hasRelevantChanges = false;

      for (let i = 0; i < mutations.length; i++) {
        const mutation = mutations[i];

        if (mutation.type === 'childList' && mutation.addedNodes.length > 0) {
          for (let j = 0; j < mutation.addedNodes.length; j++) {
            const node = mutation.addedNodes[j];
            // PazarRadar tarafindan eklenmis elemanlari atla
            if (node.nodeType === Node.ELEMENT_NODE && !node.classList.contains(PR_CLASS)) {
              hasRelevantChanges = true;
              break;
            }
          }
        }

        if (hasRelevantChanges) {
          break;
        }
      }

      if (hasRelevantChanges) {
        debouncedProcessPage();
      }
    });

    contentObserver.observe(targetNode, {
      childList: true,
      subtree: true
    });
  }

  /**
   * popstate event listener'ini baslatir.
   * Tarayici ileri/geri navigasyonlarini yakalar.
   */
  function startPopstateListener() {
    window.addEventListener('popstate', function () {
      // URL degistiyse sayfayi yeniden isle
      debouncedProcessPage();
    });
  }

  /**
   * History API'sini sarmalayarak pushState/replaceState navigasyonlarini yakalar.
   * Trendyol Next.js router'i bu metodlari kullanir.
   */
  function interceptHistoryNavigation() {
    try {
      const originalPushState = history.pushState;
      const originalReplaceState = history.replaceState;

      history.pushState = function () {
        originalPushState.apply(this, arguments);
        debouncedProcessPage();
      };

      history.replaceState = function () {
        originalReplaceState.apply(this, arguments);
        debouncedProcessPage();
      };
    } catch (_error) {
      // History API sarmalamasi basarisiz - MutationObserver yeterli olacaktir
    }
  }

  // ─── Baslangic ───────────────────────────────────────────

  /**
   * Eklentiyi baslatir.
   * Butun dinleyicileri kurar ve ilk sayfa islemesini tetikler.
   */
  function init() {
    try {
      // SPA navigasyon izlemeyi baslat
      startContentObserver();
      startPopstateListener();
      interceptHistoryNavigation();

      // Ilk sayfa islemesini performans dostu sekilde yap
      if (typeof requestIdleCallback === 'function') {
        requestIdleCallback(function () {
          processPage();
        }, { timeout: INJECTION_TIMEOUT });
      } else {
        setTimeout(processPage, 0);
      }
    } catch (error) {
      warn(' Baslangic hatasi:', error.message);
    }
  }

  // ─── Calistir ─────────────────────────────────────────────

  // Document hazir mi kontrol et (manifest.json'da run_at: document_idle oldugu icin genellikle hazirdir)
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }
})();
