/**
 * PazarRadar Commission Database
 * Comprehensive commission rate data for Turkish e-commerce marketplaces.
 *
 * Supported platforms: Trendyol, Hepsiburada, N11
 *
 * Usage (Chrome Extension content script):
 *   const db = window.PazarRadar.CommissionDB;
 *   const fees = db.calculateFees('trendyol', 'giyim_moda', 500);
 *   const comparison = db.compareAcrossPlatforms('Giyim', 500, 2);
 *
 * @namespace window.PazarRadar.CommissionDB
 * @version 1.0.0
 */
(function () {
  'use strict';

  window.PazarRadar = window.PazarRadar || {};

  // ---------------------------------------------------------------------------
  // Sabitler (Constants)
  // ---------------------------------------------------------------------------

  /** @enum {number} KDV oranlari */
  const KDV_RATES = Object.freeze({
    STANDARD: 0.20,
    REDUCED: 0.10,
    SPECIAL: 0.01
  });

  /** Varsayilan KDV orani */
  const DEFAULT_KDV_RATE = KDV_RATES.STANDARD;

  // ---------------------------------------------------------------------------
  // Platform Konfigurasyonlari
  // ---------------------------------------------------------------------------

  /**
   * Trendyol komisyon verileri
   * Komisyon KDV haric fiyat uzerinden hesaplanir.
   * Platform hizmet bedeli: 8.49 TL + KDV (her gonderi basina)
   */
  const TRENDYOL_CONFIG = Object.freeze({
    id: 'trendyol',
    name: 'Trendyol',
    commissionBase: 'kdv_exclusive',
    platformFee: 8.49,
    platformFeeKDV: true,
    categories: Object.freeze({
      elektronik_telefon: Object.freeze({
        name: 'Elektronik (Telefon)',
        rate: 0.05,
        minRate: 0.05,
        maxRate: 0.08,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'cep telefonu', 'akilli telefon', 'smartphone', 'telefon',
          'iphone', 'samsung telefon', 'android telefon', 'mobile'
        ]
      }),
      elektronik_bilgisayar: Object.freeze({
        name: 'Elektronik (Bilgisayar)',
        rate: 0.15,
        minRate: 0.12,
        maxRate: 0.20,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'bilgisayar', 'laptop', 'notebook', 'tablet', 'pc',
          'masaustu bilgisayar', 'dizustu bilgisayar', 'computer'
        ]
      }),
      elektronik_genel: Object.freeze({
        name: 'Elektronik (Genel)',
        rate: 0.10,
        minRate: 0.08,
        maxRate: 0.15,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'elektronik', 'tv', 'televizyon', 'kulaklik', 'hoparlor',
          'kamera', 'drone', 'oyun konsolu', 'monitor'
        ]
      }),
      giyim_moda: Object.freeze({
        name: 'Giyim/Moda',
        rate: 0.21,
        minRate: 0.17,
        maxRate: 0.23,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'giyim', 'moda', 'kadin giyim', 'erkek giyim', 'elbise',
          'tisort', 'gomlek', 'pantolon', 'mont', 'kaban', 'kazak',
          'etek', 'sort', 'takim elbise', 'triko', 'bluz', 'hirka',
          'jean', 'kot', 'esofman', 'pijama', 'ic giyim', 'corap'
        ]
      }),
      ayakkabi: Object.freeze({
        name: 'Ayakkabi',
        rate: 0.23,
        minRate: 0.20,
        maxRate: 0.25,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'ayakkabi', 'bot', 'cizme', 'sandalet', 'terlik',
          'sneaker', 'spor ayakkabi', 'topuklu', 'babet', 'loafer'
        ]
      }),
      canta: Object.freeze({
        name: 'Canta',
        rate: 0.21,
        minRate: 0.20,
        maxRate: 0.22,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'canta', 'el cantasi', 'sirt cantasi', 'postaci cantasi',
          'cuzdan', 'bavul', 'valiz', 'clutch', 'omuz cantasi'
        ]
      }),
      aksesuar_mucevher: Object.freeze({
        name: 'Aksesuar/Mucevher',
        rate: 0.22,
        minRate: 0.22,
        maxRate: 0.23,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'aksesuar', 'mucevher', 'taki', 'kolye', 'bileklik',
          'kupe', 'yuzuk', 'saat', 'gunes gozlugu', 'kemer',
          'sal', 'esarp', 'sapka', 'bijuteri', 'pirlanta'
        ]
      }),
      kozmetik: Object.freeze({
        name: 'Kozmetik',
        rate: 0.16,
        minRate: 0.15,
        maxRate: 0.16,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'kozmetik', 'makyaj', 'ruj', 'fondoten', 'maskara',
          'parfum', 'cilt bakim', 'sac bakim', 'guzellik',
          'serum', 'krem', 'nemlendirici', 'guneskremi'
        ]
      }),
      kisisel_bakim: Object.freeze({
        name: 'Kisisel Bakim',
        rate: 0.15,
        minRate: 0.15,
        maxRate: 0.16,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'kisisel bakim', 'dus jeli', 'sampuan', 'deodorant',
          'dis macunu', 'tiras', 'epilasyon', 'sabun', 'losyon'
        ]
      }),
      ev_yasam: Object.freeze({
        name: 'Ev & Yasam',
        rate: 0.18,
        minRate: 0.15,
        maxRate: 0.22,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'ev', 'yasam', 'mutfak', 'banyo', 'dekorasyon',
          'aydinlatma', 'hali', 'perde', 'nevresim', 'havlu',
          'tabak', 'bardak', 'tencere', 'tava', 'ev tekstil'
        ]
      }),
      mobilya: Object.freeze({
        name: 'Mobilya',
        rate: 0.195,
        minRate: 0.18,
        maxRate: 0.21,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'mobilya', 'koltuk', 'kanepe', 'yatak', 'masa',
          'sandalye', 'dolap', 'kitaplik', 'tv unitesi', 'ranza',
          'sehpa', 'konsol', 'gardrop', 'komodin'
        ]
      }),
      spor: Object.freeze({
        name: 'Spor',
        rate: 0.12,
        minRate: 0.11,
        maxRate: 0.14,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'spor', 'fitness', 'outdoor', 'kamp', 'bisiklet',
          'kosu', 'yoga', 'pilates', 'dambil', 'spor aleti',
          'spor giyim', 'forma'
        ]
      }),
      cocuk: Object.freeze({
        name: 'Cocuk',
        rate: 0.14,
        minRate: 0.10,
        maxRate: 0.15,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'cocuk', 'bebek giyim', 'cocuk giyim', 'cocuk ayakkabi',
          'bebek arabasi', 'mama sandalyesi', 'biberon', 'emzik'
        ]
      }),
      supermarket_fmcg: Object.freeze({
        name: 'Supermarket/FMCG',
        rate: 0.17,
        minRate: 0.10,
        maxRate: 0.17,
        kdvRate: KDV_RATES.REDUCED,
        aliases: [
          'supermarket', 'gida', 'icecek', 'atistirmalik',
          'temizlik', 'deterjan', 'fmcg', 'market', 'bakliyat',
          'kahve', 'cay', 'cikolata', 'bebek mamasi'
        ]
      }),
      kitap: Object.freeze({
        name: 'Kitap',
        rate: 0.15,
        minRate: 0.15,
        maxRate: 0.15,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'kitap', 'roman', 'dergi', 'kirtasiye', 'defter',
          'kalem', 'okul malzemesi'
        ]
      }),
      oyuncak: Object.freeze({
        name: 'Oyuncak',
        rate: 0.15,
        minRate: 0.14,
        maxRate: 0.15,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'oyuncak', 'lego', 'puzzle', 'bebek oyuncak', 'peluş',
          'oyun', 'masa oyunu', 'kutu oyunu'
        ]
      }),
      otomotiv: Object.freeze({
        name: 'Otomotiv',
        rate: 0.125,
        minRate: 0.10,
        maxRate: 0.15,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'otomotiv', 'yedek parca', 'araba aksesuar', 'lastik',
          'motor yagi', 'oto bakim', 'oto parfum', 'oto koltuk kilifi'
        ]
      }),
      pet: Object.freeze({
        name: 'Pet',
        rate: 0.145,
        minRate: 0.13,
        maxRate: 0.16,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'pet', 'evcil hayvan', 'kedi', 'kopek', 'kedi mamasi',
          'kopek mamasi', 'kedi kumu', 'tasma', 'akvaryum'
        ]
      })
    })
  });

  /**
   * Hepsiburada komisyon verileri
   * Komisyon KDV dahil fiyat uzerinden hesaplanir.
   */
  const HEPSIBURADA_CONFIG = Object.freeze({
    id: 'hepsiburada',
    name: 'Hepsiburada',
    commissionBase: 'kdv_inclusive',
    platformFee: 9.50,
    platformFeeKDV: true,
    categories: Object.freeze({
      altin_degerli_metal: Object.freeze({
        name: 'Altin/Degerli Metal',
        rate: 0.05,
        minRate: 0.04,
        maxRate: 0.06,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'altin', 'gumus', 'pirlanta', 'degerli metal',
          'yatirim altin', 'ceyrek altin', 'gram altin'
        ]
      }),
      elektronik_telefon: Object.freeze({
        name: 'Elektronik (Telefon)',
        rate: 0.07,
        minRate: 0.07,
        maxRate: 0.07,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'cep telefonu', 'akilli telefon', 'smartphone', 'telefon',
          'iphone', 'samsung telefon'
        ]
      }),
      elektronik_tablet: Object.freeze({
        name: 'Elektronik (Tablet)',
        rate: 0.07,
        minRate: 0.07,
        maxRate: 0.07,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'tablet', 'ipad', 'android tablet'
        ]
      }),
      tv: Object.freeze({
        name: 'TV',
        rate: 0.0678,
        minRate: 0.0678,
        maxRate: 0.0678,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'televizyon', 'tv', 'smart tv', 'led tv', 'oled tv'
        ]
      }),
      bilgisayar: Object.freeze({
        name: 'Bilgisayar',
        rate: 0.07,
        minRate: 0.07,
        maxRate: 0.17,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'bilgisayar', 'laptop', 'notebook', 'masaustu',
          'dizustu', 'pc', 'bilgisayar parcasi', 'ekran karti'
        ]
      }),
      kucuk_ev_aletleri: Object.freeze({
        name: 'Kucuk Ev Aletleri',
        rate: 0.125,
        minRate: 0.11,
        maxRate: 0.14,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'kucuk ev aleti', 'utu', 'mikser', 'blender',
          'tost makinesi', 'kahve makinesi', 'su isitici',
          'elektrik supurgesi', 'sac kurutma makinesi'
        ]
      }),
      buyuk_ev_aletleri: Object.freeze({
        name: 'Buyuk Ev Aletleri',
        rate: 0.10,
        minRate: 0.085,
        maxRate: 0.12,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'buyuk ev aleti', 'buzdolabi', 'camasir makinesi',
          'bulasik makinesi', 'firin', 'kurutma makinesi',
          'klima', 'kombi'
        ]
      }),
      bebek: Object.freeze({
        name: 'Bebek',
        rate: 0.15,
        minRate: 0.11,
        maxRate: 0.18,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'bebek', 'anne bebek', 'bebek arabasi', 'biberon', 'emzik',
          'mama sandalyesi', 'bebek bezi', 'bebek bakim', 'cocuk'
        ]
      }),
      kozmetik: Object.freeze({
        name: 'Kozmetik',
        rate: 0.16,
        minRate: 0.15,
        maxRate: 0.17,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'kozmetik', 'makyaj', 'cilt bakim', 'sac bakim',
          'guzellik', 'ruj', 'fondoten'
        ]
      }),
      parfum: Object.freeze({
        name: 'Parfum',
        rate: 0.1525,
        minRate: 0.1525,
        maxRate: 0.1525,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'parfum', 'koku', 'edp', 'edt', 'deodorant parfum'
        ]
      }),
      pet: Object.freeze({
        name: 'Pet',
        rate: 0.14,
        minRate: 0.13,
        maxRate: 0.15,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'pet', 'evcil hayvan', 'kedi mamasi', 'kopek mamasi',
          'kedi kumu', 'tasma'
        ]
      }),
      aksesuar: Object.freeze({
        name: 'Aksesuar',
        rate: 0.1832,
        minRate: 0.18,
        maxRate: 0.1864,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'aksesuar', 'saat', 'gunes gozlugu', 'kemer',
          'taki', 'mucevher', 'bijuteri'
        ]
      }),
      canta: Object.freeze({
        name: 'Canta',
        rate: 0.18,
        minRate: 0.18,
        maxRate: 0.18,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'canta', 'el cantasi', 'sirt cantasi', 'cuzdan',
          'bavul', 'valiz'
        ]
      }),
      ayakkabi: Object.freeze({
        name: 'Ayakkabi',
        rate: 0.1949,
        minRate: 0.1949,
        maxRate: 0.1949,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'ayakkabi', 'bot', 'cizme', 'sandalet', 'sneaker',
          'spor ayakkabi'
        ]
      }),
      giyim: Object.freeze({
        name: 'Giyim',
        rate: 0.18,
        minRate: 0.18,
        maxRate: 0.18,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'giyim', 'moda', 'kadin giyim', 'erkek giyim', 'elbise',
          'tisort', 'gomlek', 'pantolon', 'mont'
        ]
      }),
      ev_tekstil: Object.freeze({
        name: 'Ev Tekstil',
        rate: 0.18,
        minRate: 0.18,
        maxRate: 0.18,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'ev tekstil', 'ev yasam', 'ev dekorasyon', 'nevresim', 'havlu',
          'perde', 'yatak ortusu', 'yastik', 'yorgan', 'pike'
        ]
      }),
      oyuncak: Object.freeze({
        name: 'Oyuncak',
        rate: 0.18,
        minRate: 0.18,
        maxRate: 0.18,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'oyuncak', 'lego', 'puzzle', 'peluş', 'masa oyunu'
        ]
      }),
      mobilya: Object.freeze({
        name: 'Mobilya',
        rate: 0.20,
        minRate: 0.18,
        maxRate: 0.22,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'mobilya', 'koltuk', 'kanepe', 'yatak', 'masa',
          'sandalye', 'dolap'
        ]
      }),
      outdoor: Object.freeze({
        name: 'Outdoor',
        rate: 0.14,
        minRate: 0.14,
        maxRate: 0.14,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'outdoor', 'kamp', 'doga', 'trekking', 'cadir',
          'uyku tulumu', 'termos'
        ]
      }),
      supermarket: Object.freeze({
        name: 'Supermarket/Gida',
        rate: 0.14,
        minRate: 0.12,
        maxRate: 0.18,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'supermarket', 'gida', 'icecek', 'market', 'yiyecek',
          'atistirmalik', 'bakliyat', 'konserve'
        ]
      }),
      spor: Object.freeze({
        name: 'Spor',
        rate: 0.14,
        minRate: 0.14,
        maxRate: 0.14,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'spor', 'fitness', 'yoga', 'pilates', 'kosu',
          'spor giyim', 'spor ayakkabi', 'bisiklet'
        ]
      }),
      supermarket: Object.freeze({
        name: 'Supermarket/Gida',
        rate: 0.15,
        minRate: 0.12,
        maxRate: 0.18,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'supermarket', 'gida', 'icecek', 'market', 'atistirmalik',
          'kahvaltilik', 'bakliyat', 'konserve'
        ]
      }),
      kitap: Object.freeze({
        name: 'Kitap/Kirtasiye',
        rate: 0.15,
        minRate: 0.12,
        maxRate: 0.18,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'kitap', 'dergi', 'kirtasiye', 'defter', 'kalem',
          'roman', 'egitim', 'cocuk kitap'
        ]
      })
    })
  });

  /**
   * N11 komisyon verileri
   * Komisyon KDV haric fiyat uzerinden hesaplanir.
   */
  const N11_CONFIG = Object.freeze({
    id: 'n11',
    name: 'N11',
    commissionBase: 'kdv_exclusive',
    platformFee: 0,
    platformFeeKDV: false,
    additionalFees: Object.freeze({
      marketingFee: 0.012,    // %1.20 pazarlama hizmet bedeli
      platformPercent: 0.0067, // %0.67 pazaryeri hizmet bedeli
      withholdingTax: 0.01    // %1 stopaj
    }),
    categories: Object.freeze({
      giyim: Object.freeze({
        name: 'Giyim',
        rate: 0.20,
        minRate: 0.20,
        maxRate: 0.20,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'giyim', 'moda', 'kadin giyim', 'erkek giyim', 'elbise',
          'tisort', 'gomlek', 'pantolon', 'mont', 'kazak'
        ]
      }),
      ayakkabi_canta: Object.freeze({
        name: 'Ayakkabi/Canta',
        rate: 0.17,
        minRate: 0.15,
        maxRate: 0.20,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'ayakkabi', 'canta', 'bot', 'cizme', 'sandalet',
          'el cantasi', 'sirt cantasi', 'cuzdan', 'valiz'
        ]
      }),
      kozmetik: Object.freeze({
        name: 'Kozmetik',
        rate: 0.15,
        minRate: 0.10,
        maxRate: 0.15,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'kozmetik', 'makyaj', 'parfum', 'cilt bakim', 'sac bakim',
          'guzellik', 'kisisel bakim'
        ]
      }),
      yatirim_altin: Object.freeze({
        name: 'Yatirim Altin',
        rate: 0.05,
        minRate: 0.05,
        maxRate: 0.05,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'altin', 'yatirim altin', 'gram altin', 'ceyrek altin',
          'gumus', 'degerli metal'
        ]
      }),
      otomotiv: Object.freeze({
        name: 'Otomotiv',
        rate: 0.06,
        minRate: 0.00,
        maxRate: 0.12,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'otomotiv', 'yedek parca', 'araba aksesuar', 'lastik',
          'motor yagi', 'arac'
        ]
      }),
      spor_outdoor: Object.freeze({
        name: 'Spor/Outdoor',
        rate: 0.12,
        minRate: 0.065,
        maxRate: 0.17,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'spor', 'outdoor', 'fitness', 'kamp', 'bisiklet',
          'kosu', 'yoga', 'spor giyim'
        ]
      }),
      elektronik_telefon: Object.freeze({
        name: 'Elektronik (Telefon)',
        rate: 0.05,
        minRate: 0.05,
        maxRate: 0.10,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'cep telefonu', 'akilli telefon', 'smartphone', 'telefon',
          'iphone', 'samsung telefon'
        ]
      }),
      elektronik_bilgisayar: Object.freeze({
        name: 'Elektronik (Bilgisayar)',
        rate: 0.055,
        minRate: 0.055,
        maxRate: 0.06,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'bilgisayar', 'laptop', 'notebook', 'tablet', 'ipad', 'pc',
          'masaustu bilgisayar'
        ]
      }),
      elektronik_genel: Object.freeze({
        name: 'Elektronik (Genel)',
        rate: 0.08,
        minRate: 0.06,
        maxRate: 0.15,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'elektronik', 'tv', 'televizyon', 'kulaklik', 'hoparlor',
          'kamera', 'oyun konsolu', 'monitor'
        ]
      }),
      ev_yasam: Object.freeze({
        name: 'Ev & Yasam',
        rate: 0.15,
        minRate: 0.15,
        maxRate: 0.25,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'ev', 'yasam', 'mutfak', 'banyo', 'dekorasyon',
          'mobilya', 'ev tekstil', 'aydinlatma'
        ]
      }),
      anne_bebek: Object.freeze({
        name: 'Anne & Bebek',
        rate: 0.12,
        minRate: 0.10,
        maxRate: 0.15,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'anne bebek', 'bebek', 'bebek arabasi', 'biberon',
          'mama', 'bebek bezi', 'emzik', 'cocuk'
        ]
      }),
      oyuncak: Object.freeze({
        name: 'Oyuncak',
        rate: 0.12,
        minRate: 0.10,
        maxRate: 0.15,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'oyuncak', 'lego', 'puzzle', 'pelus', 'masa oyunu',
          'bebek oyuncak', 'aksiyon figur'
        ]
      }),
      kitap: Object.freeze({
        name: 'Kitap/Kirtasiye',
        rate: 0.10,
        minRate: 0.08,
        maxRate: 0.12,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'kitap', 'dergi', 'kirtasiye', 'defter', 'kalem',
          'roman', 'egitim'
        ]
      }),
      supermarket: Object.freeze({
        name: 'Supermarket/Gida',
        rate: 0.10,
        minRate: 0.08,
        maxRate: 0.15,
        kdvRate: KDV_RATES.STANDARD,
        aliases: [
          'supermarket', 'gida', 'icecek', 'market', 'yiyecek',
          'atistirmalik', 'bakliyat'
        ]
      })
    })
  });

  // ---------------------------------------------------------------------------
  // Tum platformlar - tek sozlukte toplandi
  // ---------------------------------------------------------------------------

  /** @type {Object<string, Readonly<Object>>} */
  const PLATFORMS = Object.freeze({
    trendyol: TRENDYOL_CONFIG,
    hepsiburada: HEPSIBURADA_CONFIG,
    n11: N11_CONFIG
  });

  // ---------------------------------------------------------------------------
  // Kargo maliyet tablosu (desi bazli, tahmini TL)
  // ---------------------------------------------------------------------------

  /**
   * Desi bazli tahmini kargo maliyetleri.
   * Her eleman [maxDesi, maliyet] ciftidir.
   * Trendyol, Hepsiburada ve N11 icin yaklasik benzer araliklar.
   *
   * @type {ReadonlyArray<Readonly<[number, number]>>}
   */
  const CARGO_COST_TABLE = Object.freeze([
    Object.freeze([1, 35]),
    Object.freeze([3, 45]),
    Object.freeze([5, 65]),
    Object.freeze([10, 85]),
    Object.freeze([15, 120]),
    Object.freeze([20, 160]),
    Object.freeze([30, 220]),
    Object.freeze([Infinity, 300])
  ]);

  // ---------------------------------------------------------------------------
  // Yardimci Fonksiyonlar (Helper Functions)
  // ---------------------------------------------------------------------------

  /**
   * Turkce karakterleri ASCII karsiliklarina donusturur.
   * Fuzzy eslestirme icin normalizasyon.
   *
   * @param {string} text - Normalize edilecek metin
   * @returns {string} Normalize edilmis metin
   */
  function normalizeTurkish(text) {
    if (typeof text !== 'string') {
      return '';
    }

    return text
      .toLowerCase()
      .replace(/ğ/g, 'g')
      .replace(/ü/g, 'u')
      .replace(/ş/g, 's')
      .replace(/ı/g, 'i')
      .replace(/ö/g, 'o')
      .replace(/ç/g, 'c')
      .replace(/İ/g, 'i')
      .replace(/Ğ/g, 'g')
      .replace(/Ü/g, 'u')
      .replace(/Ş/g, 's')
      .replace(/Ö/g, 'o')
      .replace(/Ç/g, 'c')
      .trim();
  }

  /**
   * Iki metin arasindaki benzerlik skorunu hesaplar.
   * Levenshtein mesafesi tabanli basit benzerlik olcumu +
   * kelime bazli token eslestirme.
   *
   * @param {string} source - Kaynak metin
   * @param {string} target - Hedef metin
   * @returns {number} 0-1 arasi benzerlik skoru
   */
  function similarityScore(source, target) {
    const normalizedSource = normalizeTurkish(source);
    const normalizedTarget = normalizeTurkish(target);

    if (normalizedSource === normalizedTarget) {
      return 1.0;
    }

    if (normalizedSource.length === 0 || normalizedTarget.length === 0) {
      return 0.0;
    }

    // Birebir icerir mi kontrol et - yuksek agirlikli
    if (normalizedTarget.includes(normalizedSource)) {
      return 0.85 + (0.15 * normalizedSource.length / normalizedTarget.length);
    }
    if (normalizedSource.includes(normalizedTarget)) {
      return 0.80 + (0.15 * normalizedTarget.length / normalizedSource.length);
    }

    // Token bazli eslestirme
    const sourceTokens = normalizedSource.split(/\s+/).filter(Boolean);
    const targetTokens = normalizedTarget.split(/\s+/).filter(Boolean);

    if (sourceTokens.length === 0 || targetTokens.length === 0) {
      return 0.0;
    }

    let matchCount = 0;
    for (let i = 0; i < sourceTokens.length; i++) {
      for (let j = 0; j < targetTokens.length; j++) {
        if (sourceTokens[i] === targetTokens[j]) {
          matchCount += 1;
          break;
        }
        // Kismi eslestirme: token birbirini iceriyorsa
        if (
          sourceTokens[i].length >= 3 &&
          targetTokens[j].length >= 3 &&
          (sourceTokens[i].includes(targetTokens[j]) ||
            targetTokens[j].includes(sourceTokens[i]))
        ) {
          matchCount += 0.7;
          break;
        }
      }
    }

    const maxTokens = Math.max(sourceTokens.length, targetTokens.length);
    return matchCount / maxTokens;
  }

  /**
   * Platform konfigurasyonunu dondurur.
   *
   * @param {string} platformId - Platform tanimlayicisi (trendyol, hepsiburada, n11)
   * @returns {Readonly<Object>|null} Platform konfigurasyonu veya null
   */
  function getPlatformConfig(platformId) {
    if (typeof platformId !== 'string') {
      return null;
    }

    const normalized = platformId.toLowerCase().trim();
    return PLATFORMS[normalized] || null;
  }

  /**
   * Desi bazli kargo maliyetini hesaplar.
   *
   * @param {number} desi - Kargo desi degeri
   * @returns {number} Tahmini kargo maliyeti (TL)
   */
  function estimateCargoCost(desi) {
    if (typeof desi !== 'number' || desi <= 0 || !Number.isFinite(desi)) {
      return CARGO_COST_TABLE[0][1];
    }

    for (let i = 0; i < CARGO_COST_TABLE.length; i++) {
      if (desi <= CARGO_COST_TABLE[i][0]) {
        return CARGO_COST_TABLE[i][1];
      }
    }

    return CARGO_COST_TABLE[CARGO_COST_TABLE.length - 1][1];
  }

  /**
   * Sayiyi iki ondalik basamaga yuvarlar.
   *
   * @param {number} value - Yuvarlanacak sayi
   * @returns {number} Yuvarlanmis sayi
   */
  function roundTwo(value) {
    return Math.round(value * 100) / 100;
  }

  // ---------------------------------------------------------------------------
  // Ana API
  // ---------------------------------------------------------------------------

  /**
   * PazarRadar Komisyon Veritabani.
   * Turk e-ticaret pazaryerleri icin komisyon oranlari,
   * ucret hesaplama ve platformlar arasi karsilastirma sunar.
   *
   * @namespace
   */
  const CommissionDB = Object.freeze({
    /**
     * Desteklenen platform listesini dondurur.
     *
     * @returns {ReadonlyArray<{id: string, name: string}>}
     */
    getPlatforms: function getPlatforms() {
      return Object.keys(PLATFORMS).map(function (key) {
        return Object.freeze({ id: key, name: PLATFORMS[key].name });
      });
    },

    /**
     * Belirli bir platform ve kategori icin komisyon oranini dondurur.
     *
     * @param {string} platform - Platform tanimlayicisi (trendyol, hepsiburada, n11)
     * @param {string} categoryKey - Kategori anahtari (ornegin: giyim_moda, elektronik_telefon)
     * @returns {{rate: number, minRate: number, maxRate: number, platformFee: number, platformFeeKDV: boolean, kdvRate: number, commissionBase: string}|null}
     *   Komisyon bilgisi veya null (bulunamazsa)
     */
    getRate: function getRate(platform, categoryKey) {
      const config = getPlatformConfig(platform);
      if (!config) {
        return null;
      }

      const category = config.categories[categoryKey];
      if (!category) {
        return null;
      }

      return Object.freeze({
        rate: category.rate,
        minRate: category.minRate,
        maxRate: category.maxRate,
        platformFee: config.platformFee,
        platformFeeKDV: config.platformFeeKDV,
        kdvRate: category.kdvRate,
        commissionBase: config.commissionBase
      });
    },

    /**
     * Belirli bir platformun tum kategorilerini dondurur.
     *
     * @param {string} platform - Platform tanimlayicisi
     * @returns {ReadonlyArray<{key: string, name: string, rate: number, minRate: number, maxRate: number}>|null}
     *   Kategori listesi veya null
     */
    getCategories: function getCategories(platform) {
      const config = getPlatformConfig(platform);
      if (!config) {
        return null;
      }

      const keys = Object.keys(config.categories);
      return Object.freeze(
        keys.map(function (key) {
          const cat = config.categories[key];
          return Object.freeze({
            key: key,
            name: cat.name,
            rate: cat.rate,
            minRate: cat.minRate,
            maxRate: cat.maxRate
          });
        })
      );
    },

    /**
     * Kategori adina gore en iyi eslesen kategoriyi bulur (fuzzy match).
     * Turkce karakter duyarli.
     *
     * @param {string} platform - Platform tanimlayicisi
     * @param {string} categoryName - Aranacak kategori adi (ornegin: "Cep Telefonu", "Kadin Elbise")
     * @returns {{key: string, name: string, rate: number, score: number}|null}
     *   En iyi eslesen kategori veya null
     */
    findCategory: function findCategory(platform, categoryName) {
      const config = getPlatformConfig(platform);
      if (!config || typeof categoryName !== 'string' || categoryName.trim().length === 0) {
        return null;
      }

      const normalizedInput = normalizeTurkish(categoryName);
      let bestMatch = null;
      let bestScore = 0;

      const categoryKeys = Object.keys(config.categories);

      for (let i = 0; i < categoryKeys.length; i++) {
        const key = categoryKeys[i];
        const cat = config.categories[key];

        // Kategori adiyla eslestirme skoru
        const nameScore = similarityScore(normalizedInput, cat.name);

        // Alias'larla eslestirme - en yuksek skoru al
        let aliasScore = 0;
        if (cat.aliases && cat.aliases.length > 0) {
          for (let j = 0; j < cat.aliases.length; j++) {
            const score = similarityScore(normalizedInput, cat.aliases[j]);
            if (score > aliasScore) {
              aliasScore = score;
            }
          }
        }

        // Key ile de eslestir (underscore'lari bosluga cevir)
        const keyAsName = key.replace(/_/g, ' ');
        const keyScore = similarityScore(normalizedInput, keyAsName);

        // En yuksek skoru sec
        const finalScore = Math.max(nameScore, aliasScore, keyScore);

        if (finalScore > bestScore) {
          bestScore = finalScore;
          bestMatch = {
            key: key,
            name: cat.name,
            rate: cat.rate,
            score: roundTwo(finalScore)
          };
        }
      }

      // Minimum esik: cok dusuk skorlu eslesmeleri reddet
      if (bestScore < 0.25) {
        return null;
      }

      return bestMatch !== null ? Object.freeze(bestMatch) : null;
    },

    /**
     * Bir satis icin tum ucretleri hesaplar.
     *
     * Trendyol: Komisyon KDV haric fiyat uzerinden.
     *   KDV haric fiyat = satisFiyati / (1 + kdvOrani)
     *   Komisyon = kdvHaricFiyat * komisyonOrani
     *   Komisyon KDV = komisyon * 0.20
     *   Platform Hizmet Bedeli = 8.49 TL
     *   Platform Hizmet Bedeli KDV = 8.49 * 0.20
     *
     * Hepsiburada: Komisyon KDV dahil fiyat uzerinden.
     *   Komisyon = satisFiyati * komisyonOrani
     *   Komisyon KDV = komisyon * 0.20
     *
     * N11: Komisyon KDV haric fiyat uzerinden.
     *   KDV haric fiyat = satisFiyati / (1 + kdvOrani)
     *   Komisyon = kdvHaricFiyat * komisyonOrani
     *   Komisyon KDV = komisyon * 0.20
     *
     * @param {string} platform - Platform tanimlayicisi
     * @param {string} categoryKey - Kategori anahtari
     * @param {number} salePrice - Satis fiyati (KDV dahil, TL)
     * @param {number} [cargoWeight=1] - Kargo desi degeri
     * @returns {{
     *   salePrice: number,
     *   salePriceExKDV: number,
     *   kdvRate: number,
     *   kdvAmount: number,
     *   commissionBase: string,
     *   commissionableAmount: number,
     *   commissionRate: number,
     *   commission: number,
     *   commissionKDV: number,
     *   platformFee: number,
     *   platformFeeKDV: number,
     *   estimatedCargo: number,
     *   totalFees: number,
     *   netRevenue: number,
     *   netMarginPercent: number
     * }|null} Hesaplama sonucu veya null
     */
    calculateFees: function calculateFees(platform, categoryKey, salePrice, cargoWeight) {
      const config = getPlatformConfig(platform);
      if (!config) {
        return null;
      }

      const category = config.categories[categoryKey];
      if (!category) {
        return null;
      }

      if (typeof salePrice !== 'number' || salePrice <= 0 || !Number.isFinite(salePrice)) {
        return null;
      }

      const effectiveCargoWeight =
        typeof cargoWeight === 'number' && cargoWeight > 0 && Number.isFinite(cargoWeight)
          ? cargoWeight
          : 1;

      const kdvRate = category.kdvRate || DEFAULT_KDV_RATE;
      const salePriceExKDV = roundTwo(salePrice / (1 + kdvRate));
      const kdvAmount = roundTwo(salePrice - salePriceExKDV);

      // Komisyon hesabi: platformun komisyon bazina gore
      let commissionableAmount;
      if (config.commissionBase === 'kdv_inclusive') {
        commissionableAmount = salePrice;
      } else {
        commissionableAmount = salePriceExKDV;
      }

      const commissionRate = category.rate;
      const commission = roundTwo(commissionableAmount * commissionRate);
      // Komisyon uzerinden KDV (%20 sabit)
      const commissionKDV = roundTwo(commission * KDV_RATES.STANDARD);

      // Platform hizmet bedeli
      const platformFee = config.platformFee || 0;
      const platformFeeKDV = config.platformFeeKDV
        ? roundTwo(platformFee * KDV_RATES.STANDARD)
        : 0;

      // Tahmini kargo
      const cargo = estimateCargoCost(effectiveCargoWeight);

      // N11 ek ucretler (pazarlama + platform + stopaj)
      let additionalFeesTotal = 0;
      if (config.additionalFees) {
        const af = config.additionalFees;
        const mktFee = roundTwo(salePrice * (af.marketingFee || 0));
        const pltFee = roundTwo(salePrice * (af.platformPercent || 0));
        const whTax = roundTwo(salePrice * (af.withholdingTax || 0));
        additionalFeesTotal = roundTwo(mktFee + pltFee + whTax);
      }

      // Toplam kesintiler
      const totalFees = roundTwo(
        commission + commissionKDV + platformFee + platformFeeKDV + cargo + additionalFeesTotal
      );

      // Net gelir
      const netRevenue = roundTwo(salePrice - totalFees);

      // Net kar marji (%)
      const netMarginPercent = roundTwo((netRevenue / salePrice) * 100);

      return Object.freeze({
        salePrice: salePrice,
        salePriceExKDV: salePriceExKDV,
        kdvRate: kdvRate,
        kdvAmount: kdvAmount,
        commissionBase: config.commissionBase,
        commissionableAmount: commissionableAmount,
        commissionRate: commissionRate,
        commission: commission,
        commissionKDV: commissionKDV,
        platformFee: platformFee,
        platformFeeKDV: platformFeeKDV,
        additionalFees: additionalFeesTotal,
        estimatedCargo: cargo,
        totalFees: totalFees,
        netRevenue: netRevenue,
        netMarginPercent: netMarginPercent
      });
    },

    /**
     * Birden fazla platform arasinda fiyat/komisyon karsilastirmasi yapar.
     * Verilen kategori adina gore tum platformlarda eslestirme yapar
     * ve sonuclari net gelire gore siralar.
     *
     * @param {string} categoryName - Kategori adi (fuzzy match uygulanir)
     * @param {number} salePrice - Satis fiyati (KDV dahil, TL)
     * @param {number} [cargoWeight=1] - Kargo desi degeri
     * @returns {ReadonlyArray<{
     *   platform: string,
     *   platformName: string,
     *   matchedCategory: string,
     *   matchScore: number,
     *   netRevenue: number,
     *   netMarginPercent: number,
     *   totalFees: number,
     *   breakdown: Object
     * }>} Net gelire gore azalan sirada siralanmis karsilastirma sonuclari
     */
    compareAcrossPlatforms: function compareAcrossPlatforms(categoryName, salePrice, cargoWeight) {
      if (
        typeof categoryName !== 'string' ||
        categoryName.trim().length === 0 ||
        typeof salePrice !== 'number' ||
        salePrice <= 0 ||
        !Number.isFinite(salePrice)
      ) {
        return Object.freeze([]);
      }

      const effectiveCargoWeight =
        typeof cargoWeight === 'number' && cargoWeight > 0 && Number.isFinite(cargoWeight)
          ? cargoWeight
          : 1;

      var results = [];
      var platformIds = Object.keys(PLATFORMS);

      for (var p = 0; p < platformIds.length; p++) {
        var platformId = platformIds[p];
        var match = CommissionDB.findCategory(platformId, categoryName);

        if (match === null || match.score < 0.35) {
          continue;
        }

        var fees = CommissionDB.calculateFees(
          platformId,
          match.key,
          salePrice,
          effectiveCargoWeight
        );

        if (fees === null) {
          continue;
        }

        results.push({
          platform: platformId,
          platformName: PLATFORMS[platformId].name,
          matchedCategory: match.name,
          matchScore: match.score,
          netRevenue: fees.netRevenue,
          netMarginPercent: fees.netMarginPercent,
          totalFees: fees.totalFees,
          breakdown: fees
        });
      }

      // Net gelire gore azalan sirala
      results.sort(function (a, b) {
        return b.netRevenue - a.netRevenue;
      });

      return Object.freeze(
        results.map(function (item) {
          return Object.freeze(item);
        })
      );
    },

    /**
     * Komisyon oranlarinin min/max araliginda belirli bir oranla hesaplama yapar.
     * Saticilarin "en iyi" ve "en kotu" senaryo analizleri icin.
     *
     * @param {string} platform - Platform tanimlayicisi
     * @param {string} categoryKey - Kategori anahtari
     * @param {number} salePrice - Satis fiyati (KDV dahil, TL)
     * @param {number} [cargoWeight=1] - Kargo desi degeri
     * @returns {{
     *   bestCase: Object,
     *   worstCase: Object,
     *   typical: Object
     * }|null} Senaryo analizi veya null
     */
    calculateScenarios: function calculateScenarios(platform, categoryKey, salePrice, cargoWeight) {
      var config = getPlatformConfig(platform);
      if (!config) {
        return null;
      }

      var category = config.categories[categoryKey];
      if (!category) {
        return null;
      }

      if (typeof salePrice !== 'number' || salePrice <= 0 || !Number.isFinite(salePrice)) {
        return null;
      }

      var effectiveCargoWeight =
        typeof cargoWeight === 'number' && cargoWeight > 0 && Number.isFinite(cargoWeight)
          ? cargoWeight
          : 1;

      // Gecici olarak farkli oranlarla hesapla
      // bestCase: minimum komisyon orani
      // worstCase: maksimum komisyon orani
      // typical: ortalama (varsayilan) oran

      /**
       * Verilen oran ile hesaplama yapan yardimci fonksiyon.
       * Ana calculateFees fonksiyonunun mantigini yeniden uygular
       * ama belirli bir oran kullanir.
       */
      function calcWithRate(rate) {
        var kdvRate = category.kdvRate || DEFAULT_KDV_RATE;
        var salePriceExKDV = roundTwo(salePrice / (1 + kdvRate));
        var kdvAmountCalc = roundTwo(salePrice - salePriceExKDV);

        var commissionableAmount =
          config.commissionBase === 'kdv_inclusive' ? salePrice : salePriceExKDV;

        var commission = roundTwo(commissionableAmount * rate);
        var commissionKDV = roundTwo(commission * KDV_RATES.STANDARD);
        var platformFee = config.platformFee || 0;
        var platformFeeKDV = config.platformFeeKDV
          ? roundTwo(platformFee * KDV_RATES.STANDARD)
          : 0;
        var cargo = estimateCargoCost(effectiveCargoWeight);
        var additionalFeesTotal = 0;
        if (config.additionalFees) {
          var af = config.additionalFees;
          additionalFeesTotal = roundTwo(
            salePrice * ((af.marketingFee || 0) + (af.platformPercent || 0) + (af.withholdingTax || 0))
          );
        }
        var totalFees = roundTwo(
          commission + commissionKDV + platformFee + platformFeeKDV + cargo + additionalFeesTotal
        );
        var netRevenue = roundTwo(salePrice - totalFees);
        var netMarginPercent = roundTwo((netRevenue / salePrice) * 100);

        return Object.freeze({
          salePrice: salePrice,
          salePriceExKDV: salePriceExKDV,
          kdvRate: kdvRate,
          kdvAmount: kdvAmountCalc,
          commissionRate: rate,
          commission: commission,
          commissionKDV: commissionKDV,
          platformFee: platformFee,
          platformFeeKDV: platformFeeKDV,
          additionalFees: additionalFeesTotal,
          estimatedCargo: cargo,
          totalFees: totalFees,
          netRevenue: netRevenue,
          netMarginPercent: netMarginPercent
        });
      }

      return Object.freeze({
        bestCase: calcWithRate(category.minRate),
        worstCase: calcWithRate(category.maxRate),
        typical: calcWithRate(category.rate)
      });
    },

    /**
     * Kargo maliyet tablosunu dondurur.
     *
     * @returns {ReadonlyArray<{maxDesi: number|string, cost: number}>}
     */
    getCargoTable: function getCargoTable() {
      return Object.freeze(
        CARGO_COST_TABLE.map(function (entry) {
          return Object.freeze({
            maxDesi: entry[0] === Infinity ? '30+' : entry[0],
            cost: entry[1]
          });
        })
      );
    },

    /**
     * Belirli bir desi icin tahmini kargo maliyetini dondurur.
     *
     * @param {number} desi - Kargo desi degeri
     * @returns {number} Tahmini kargo maliyeti (TL)
     */
    estimateCargo: function estimateCargoPublic(desi) {
      return estimateCargoCost(desi);
    },

    /**
     * Belirli bir fiyat ve platformda "break-even" (basabas) maliyet fiyatini hesaplar.
     * Bu fiyatin altinda satici zarar eder.
     *
     * @param {string} platform - Platform tanimlayicisi
     * @param {string} categoryKey - Kategori anahtari
     * @param {number} salePrice - Satis fiyati (KDV dahil, TL)
     * @param {number} [cargoWeight=1] - Kargo desi degeri
     * @returns {{breakEvenCost: number, totalFees: number, maxProductCost: number}|null}
     */
    getBreakEvenCost: function getBreakEvenCost(platform, categoryKey, salePrice, cargoWeight) {
      var fees = CommissionDB.calculateFees(platform, categoryKey, salePrice, cargoWeight);
      if (fees === null) {
        return null;
      }

      return Object.freeze({
        breakEvenCost: fees.netRevenue,
        totalFees: fees.totalFees,
        maxProductCost: fees.netRevenue
      });
    },

    /**
     * Hedef kar marjina ulasmak icin gereken minimum satis fiyatini hesaplar.
     *
     * @param {string} platform - Platform tanimlayicisi
     * @param {string} categoryKey - Kategori anahtari
     * @param {number} productCost - Urun maliyeti (TL)
     * @param {number} targetMarginPercent - Hedef kar marji (ornegin 20 = %20)
     * @param {number} [cargoWeight=1] - Kargo desi degeri
     * @returns {{suggestedPrice: number, netRevenue: number, actualMarginPercent: number, breakdown: Object}|null}
     */
    suggestPrice: function suggestPrice(platform, categoryKey, productCost, targetMarginPercent, cargoWeight) {
      var config = getPlatformConfig(platform);
      if (!config) {
        return null;
      }

      var category = config.categories[categoryKey];
      if (!category) {
        return null;
      }

      if (
        typeof productCost !== 'number' || productCost <= 0 ||
        typeof targetMarginPercent !== 'number' || targetMarginPercent < 0 ||
        !Number.isFinite(productCost) || !Number.isFinite(targetMarginPercent)
      ) {
        return null;
      }

      var effectiveCargoWeight =
        typeof cargoWeight === 'number' && cargoWeight > 0 && Number.isFinite(cargoWeight)
          ? cargoWeight
          : 1;

      // Iteratif yaklasim: fiyati kademeli olarak artirarak hedef marji bulma
      // Baslangiç tahmini: maliyet / (1 - toplam kesinti orani tahmini)
      var kdvRate = category.kdvRate || DEFAULT_KDV_RATE;
      var commissionRate = category.rate;
      var platformFee = config.platformFee || 0;
      var platformFeeKDV = config.platformFeeKDV
        ? roundTwo(platformFee * KDV_RATES.STANDARD)
        : 0;
      var cargo = estimateCargoCost(effectiveCargoWeight);
      var fixedCosts = platformFee + platformFeeKDV + cargo;

      // Hedef: netRevenue = productCost * (1 + targetMarginPercent/100)
      var targetNetRevenue = productCost * (1 + targetMarginPercent / 100);

      // Baslangic tahmini
      var estimatedEffectiveRate;
      if (config.commissionBase === 'kdv_inclusive') {
        // Komisyon KDV dahil fiyat uzerinden: totalRate = rate * 1.20 (komisyon + komisyon KDV)
        estimatedEffectiveRate = commissionRate * (1 + KDV_RATES.STANDARD);
      } else {
        // Komisyon KDV haric fiyat uzerinden
        estimatedEffectiveRate =
          (commissionRate * (1 + KDV_RATES.STANDARD)) / (1 + kdvRate);
      }

      var suggestedPrice = roundTwo(
        (targetNetRevenue + fixedCosts) / (1 - estimatedEffectiveRate)
      );

      // 10 iterasyon ile hassasiyeti artir
      for (var iter = 0; iter < 10; iter++) {
        var testFees = CommissionDB.calculateFees(platform, categoryKey, suggestedPrice, effectiveCargoWeight);
        if (testFees === null) {
          return null;
        }

        var diff = testFees.netRevenue - targetNetRevenue;
        if (Math.abs(diff) < 0.5) {
          break;
        }

        // Fark kadar fiyati ayarla
        suggestedPrice = roundTwo(suggestedPrice - diff + (diff > 0 ? 0 : Math.abs(diff) * 0.1));
      }

      // Son hesaplama
      var finalFees = CommissionDB.calculateFees(platform, categoryKey, suggestedPrice, effectiveCargoWeight);
      if (finalFees === null) {
        return null;
      }

      var actualMargin = roundTwo(((finalFees.netRevenue - productCost) / productCost) * 100);

      return Object.freeze({
        suggestedPrice: suggestedPrice,
        netRevenue: finalFees.netRevenue,
        actualMarginPercent: actualMargin,
        productCost: productCost,
        targetMarginPercent: targetMarginPercent,
        breakdown: finalFees
      });
    },

    /** Modul versiyonu */
    VERSION: '1.0.0'
  });

  // Namespace'e ata
  window.PazarRadar.CommissionDB = CommissionDB;
})();
