/**
 * PazarRadar Sales Estimation Engine
 *
 * Turkiye e-ticaret pazaryerleri (Trendyol, Hepsiburada, N11) icin
 * satis hacmi tahmin motoru. Urunlerin acik satis verisi olmadigi icin
 * yorum sayisi, favori, stok degisimi gibi sinyallerden tahmin uretir.
 *
 * @namespace window.PazarRadar.SalesEstimator
 */
(function () {
  'use strict';

  window.PazarRadar = window.PazarRadar || {};

  // ---------------------------------------------------------------------------
  // Sabitler
  // ---------------------------------------------------------------------------

  /** @type {string} chrome.storage key */
  const STORAGE_KEY = 'stock_tracking';

  /** Maksimum takip edilebilecek urun sayisi */
  const MAX_TRACKED_PRODUCTS = 500;

  /** Takip verisi saklama suresi (gun) */
  const TRACKING_RETENTION_DAYS = 30;

  /**
   * Kategori bazli yorum carpani.
   * Ornegin electronics: 20 => her 20 alicidan 1'i yorum birakir.
   * @type {Record<string, number>}
   */
  const REVIEW_MULTIPLIERS = Object.freeze({
    electronics: 20,
    fashion: 12,
    home: 15,
    beauty: 15,
    food: 8,
    toys: 18,
    sports: 15,
    default: 15,
  });

  /**
   * Kategori bazli favori-donusum orani.
   * Ornegin fashion: 0.08 => favoriye ekleyenlerin %8'i ayda satin alir.
   * @type {Record<string, number>}
   */
  const FAVORITE_CONVERSIONS = Object.freeze({
    fashion: 0.08,
    electronics: 0.05,
    home: 0.06,
    beauty: 0.07,
    food: 0.04,
    toys: 0.06,
    sports: 0.05,
    default: 0.06,
  });

  /**
   * Talep seviyesi esik degerleri (aylik tahmini satis adetine gore).
   * @type {Array<{threshold: number, level: string}>}
   */
  const DEMAND_THRESHOLDS = Object.freeze([
    { threshold: 500, level: 'very_high' },
    { threshold: 200, level: 'high' },
    { threshold: 50, level: 'medium' },
    { threshold: 10, level: 'low' },
    { threshold: 0, level: 'very_low' },
  ]);

  /**
   * Talep seviyesi Turkce aciklamalari.
   * @type {Record<string, string>}
   */
  const DEMAND_LABELS = Object.freeze({
    very_high: 'Cok Yuksek Talep',
    high: 'Yuksek Talep',
    medium: 'Orta Talep',
    low: 'Dusuk Talep',
    very_low: 'Cok Dusuk Talep',
  });

  /**
   * Nis degerlendirmesi sonuc etiketleri.
   * @type {Record<string, {text: string, color: string}>}
   */
  const VERDICT_MAP = Object.freeze({
    excellent: { text: 'Mukemmel Firsat', color: '#16a34a' },
    good: { text: 'Girilmesi Onerilen Kategori', color: '#22c55e' },
    moderate: { text: 'Orta Riskli Kategori', color: '#f97316' },
    risky: { text: 'Riskli Kategori', color: '#ef4444' },
    avoid: { text: 'Kacinilmasi Gereken Kategori', color: '#991b1b' },
  });

  // ---------------------------------------------------------------------------
  // Yardimci Fonksiyonlar
  // ---------------------------------------------------------------------------

  /**
   * Degeri belirli bir araliga normalize eder (1-10 arasi).
   * @param {number} value - Normalize edilecek deger
   * @param {number} min   - Kaynak aralik alt siniri
   * @param {number} max   - Kaynak aralik ust siniri
   * @returns {number} 1-10 arasi normalize edilmis deger
   */
  function normalize(value, min, max) {
    if (max <= min) return 5;
    const clamped = Math.max(min, Math.min(max, value));
    return 1 + ((clamped - min) / (max - min)) * 9;
  }

  /**
   * Sayi degerini Turkce gosterim formatina cevirir.
   * Ornek: 134500 => "134K", 2300000 => "2.3M"
   * @param {number} value
   * @returns {string}
   */
  function formatTurkishNumber(value) {
    if (value == null || Number.isNaN(value)) return '0';

    const abs = Math.abs(value);

    if (abs >= 1_000_000) {
      const millions = value / 1_000_000;
      // Tam sayi ise ondalik gosterme
      return millions % 1 === 0
        ? `${millions}M`
        : `${millions.toFixed(1).replace('.0', '')}M`;
    }

    if (abs >= 1_000) {
      const thousands = value / 1_000;
      return thousands % 1 === 0
        ? `${thousands}K`
        : `${thousands.toFixed(1).replace('.0', '')}K`;
    }

    return String(Math.round(value));
  }

  /**
   * Aylik satis tahmini icin gosterim metni olusturur.
   * Guven duzeyi dusukse aralik gosterir, yuksekse yaklasik deger.
   * @param {number} sales
   * @param {string} confidence - 'low' | 'medium' | 'high'
   * @returns {string} Ornek: "~450/ay" veya "150-300/ay"
   */
  function formatSalesText(sales, confidence) {
    if (sales <= 0) return '0/ay';

    const rounded = Math.round(sales);

    if (confidence === 'high') {
      return `~${rounded}/ay`;
    }

    if (confidence === 'medium') {
      const lower = Math.round(sales * 0.7);
      const upper = Math.round(sales * 1.3);
      return `${lower}-${upper}/ay`;
    }

    // low confidence - daha genis aralik
    const lower = Math.round(sales * 0.5);
    const upper = Math.round(sales * 1.8);
    return `${lower}-${upper}/ay`;
  }

  /**
   * Aylik gelir tahmini icin Turkce gosterim metni.
   * @param {number} revenue
   * @returns {string} Ornek: "~134K TL/ay"
   */
  function formatRevenueText(revenue) {
    if (revenue <= 0) return '0 TL/ay';
    return `~${formatTurkishNumber(revenue)} TL/ay`;
  }

  /**
   * Verilen degerden kategori anahtarini cikarir.
   * Gelen categoryKey degerini bilinen kategorilere esler.
   * @param {string|undefined} categoryKey
   * @returns {string} Bilinen kategori ya da 'default'
   */
  function resolveCategory(categoryKey) {
    if (!categoryKey) return 'default';

    const key = String(categoryKey).toLowerCase().trim();

    // Turkce ve Ingilizce eslestirmeler
    const categoryMap = {
      elektronik: 'electronics',
      electronics: 'electronics',
      telefon: 'electronics',
      bilgisayar: 'electronics',
      tablet: 'electronics',
      tv: 'electronics',
      // Moda
      moda: 'fashion',
      fashion: 'fashion',
      giyim: 'fashion',
      ayakkabi: 'fashion',
      canta: 'fashion',
      aksesuar: 'fashion',
      // Ev & Yasam
      'ev': 'home',
      home: 'home',
      mobilya: 'home',
      dekorasyon: 'home',
      mutfak: 'home',
      banyo: 'home',
      // Kozmetik
      kozmetik: 'beauty',
      beauty: 'beauty',
      'kisisel bakim': 'beauty',
      parfum: 'beauty',
      // Gida
      gida: 'food',
      food: 'food',
      icecek: 'food',
      // Oyuncak
      oyuncak: 'toys',
      toys: 'toys',
      bebek: 'toys',
      // Spor
      spor: 'sports',
      sports: 'sports',
      outdoor: 'sports',
    };

    // Tam eslestirme
    if (categoryMap[key]) return categoryMap[key];

    // Kismen eslestirme (kategori adi icinde arama)
    for (const [pattern, category] of Object.entries(categoryMap)) {
      if (key.includes(pattern) || pattern.includes(key)) {
        return category;
      }
    }

    return 'default';
  }

  /**
   * Guvenli bir sekilde sayisal deger okur; gecersizse fallback dondurur.
   * @param {*} value
   * @param {number} fallback
   * @returns {number}
   */
  function safeNumber(value, fallback) {
    if (value == null) return fallback;
    const num = Number(value);
    return Number.isFinite(num) ? num : fallback;
  }

  /**
   * Talep seviyesini aylik satis adetine gore belirler.
   * @param {number} monthlySales
   * @returns {string}
   */
  function determineDemandLevel(monthlySales) {
    for (const { threshold, level } of DEMAND_THRESHOLDS) {
      if (monthlySales >= threshold) return level;
    }
    return 'very_low';
  }

  /**
   * Rekabet seviyesini satici sayisina gore belirler.
   * @param {number|undefined} totalSellers
   * @returns {string}
   */
  function determineCompetitionLevel(totalSellers) {
    if (totalSellers == null) return 'medium';
    if (totalSellers > 100) return 'high';
    if (totalSellers > 30) return 'medium';
    return 'low';
  }

  /**
   * Yorum hizi trendini hesaplar.
   * reviewsLast30Days ile gecmis ortalama karsilastirilir.
   * @param {object} signals
   * @returns {string} 'rising' | 'stable' | 'declining' | 'unknown'
   */
  function determineTrend(signals) {
    const recentReviews = safeNumber(signals.reviewsLast30Days, -1);
    const totalReviews = safeNumber(signals.totalReviews, 0);
    const listingAge = safeNumber(signals.listingAge, 0);

    // Son 30 gunluk yorum verisi yoksa bilinmiyor
    if (recentReviews < 0) return 'unknown';
    if (totalReviews <= 0 || listingAge <= 30) return 'unknown';

    // Gecmis aylik ortalama yorum
    const historicalMonthlyAvg = totalReviews / (listingAge / 30);

    if (historicalMonthlyAvg <= 0) return 'unknown';

    const ratio = recentReviews / historicalMonthlyAvg;

    if (ratio > 1.3) return 'rising';
    if (ratio < 0.7) return 'declining';
    return 'stable';
  }

  // ---------------------------------------------------------------------------
  // Tahmin Motorlari
  // ---------------------------------------------------------------------------

  /**
   * Yorum sayisindan aylik satis tahmini hesaplar.
   * @param {object} signals
   * @param {string} category - Cozumlenmis kategori anahtari
   * @returns {number} Tahmini aylik satis adedi
   */
  function estimateFromReviews(signals, category) {
    const multiplier = REVIEW_MULTIPLIERS[category] || REVIEW_MULTIPLIERS.default;
    const recentReviews = safeNumber(signals.reviewsLast30Days, -1);

    // Son 30 gun yorum verisi varsa en hassas tahmin
    if (recentReviews >= 0) {
      return recentReviews * multiplier;
    }

    // Sadece toplam yorum ve listeleme yasi varsa
    const totalReviews = safeNumber(signals.totalReviews, 0);
    const listingAge = safeNumber(signals.listingAge, 0);

    if (totalReviews > 0 && listingAge > 0) {
      const monthlyReviews = totalReviews / (listingAge / 30);
      return monthlyReviews * multiplier;
    }

    // Sadece toplam yorum varsa, varsayilan 6 aylik listeleme uzerinden hesapla
    if (totalReviews > 0) {
      const assumedMonths = 6;
      const monthlyReviews = totalReviews / assumedMonths;
      return monthlyReviews * multiplier;
    }

    return 0;
  }

  /**
   * Favori sayisindan aylik satis tahmini hesaplar.
   * @param {object} signals
   * @param {string} category - Cozumlenmis kategori anahtari
   * @returns {number} Tahmini aylik satis adedi
   */
  function estimateFromFavorites(signals, category) {
    const favoriteCount = safeNumber(signals.favoriteCount, 0);
    if (favoriteCount <= 0) return 0;

    const conversion =
      FAVORITE_CONVERSIONS[category] || FAVORITE_CONVERSIONS.default;
    return favoriteCount * conversion;
  }

  /**
   * Stok degisiminden aylik satis tahmini hesaplar. En dogru yontem.
   * @param {object} signals
   * @returns {number|null} Tahmini aylik satis adedi veya veri yoksa null
   */
  function estimateFromStock(signals) {
    const stockChange = safeNumber(signals.stockChange, null);
    const trackingDays = safeNumber(signals.trackingDays, 0);

    if (stockChange == null || trackingDays <= 0) return null;

    const dailySales = Math.abs(stockChange) / trackingDays;
    return dailySales * 30;
  }

  /**
   * Agirlikli birlesik tahmin hesaplar.
   * Farkli sinyal kaynaklarini guven agirliklariyla birlestirir.
   *
   * @param {number} reviewBased
   * @param {number} favoriteBased
   * @param {number|null} stockBased
   * @returns {{combined: number, confidence: string}}
   */
  function computeWeightedEstimate(reviewBased, favoriteBased, stockBased) {
    // Stok verisi varsa en guvenilir kaynaktir
    if (stockBased != null && stockBased > 0) {
      const combined =
        stockBased * 0.5 + reviewBased * 0.3 + favoriteBased * 0.2;
      return { combined: Math.max(0, combined), confidence: 'high' };
    }

    // Her iki sinyal de varsa orta guven
    if (reviewBased > 0 && favoriteBased > 0) {
      const combined = reviewBased * 0.5 + favoriteBased * 0.5;
      return { combined: Math.max(0, combined), confidence: 'medium' };
    }

    // Sadece yorum bazli tahmin
    if (reviewBased > 0) {
      return { combined: Math.max(0, reviewBased), confidence: 'low' };
    }

    // Sadece favori bazli tahmin
    if (favoriteBased > 0) {
      return { combined: Math.max(0, favoriteBased), confidence: 'low' };
    }

    return { combined: 0, confidence: 'low' };
  }

  // ---------------------------------------------------------------------------
  // Stok Takip Fonksiyonlari
  // ---------------------------------------------------------------------------

  /**
   * chrome.storage.local'dan stok takip verisini okur.
   * @returns {Promise<Record<string, Array<{date: number, stock: number, platform: string}>>>}
   */
  async function loadTrackingData() {
    try {
      // chrome.storage API kontrolu
      if (
        typeof chrome === 'undefined' ||
        !chrome.storage ||
        !chrome.storage.local
      ) {
        return {};
      }

      const result = await chrome.storage.local.get(STORAGE_KEY);
      return result[STORAGE_KEY] || {};
    } catch (_err) {
      return {};
    }
  }

  /**
   * Stok takip verisini chrome.storage.local'a yazar.
   * @param {Record<string, Array<{date: number, stock: number, platform: string}>>} data
   * @returns {Promise<void>}
   */
  async function saveTrackingData(data) {
    try {
      if (
        typeof chrome === 'undefined' ||
        !chrome.storage ||
        !chrome.storage.local
      ) {
        return;
      }

      await chrome.storage.local.set({ [STORAGE_KEY]: data });
    } catch (_err) {
      // Sessizce basarisiz ol - icerik betigi ortaminda hata log'lama
    }
  }

  /**
   * Eski takip kayitlarini temizler (30 gun oncesi) ve
   * toplam takip edilen urun sayisini MAX_TRACKED_PRODUCTS ile sinirlar.
   *
   * @param {Record<string, Array<{date: number, stock: number, platform: string}>>} data
   * @returns {Record<string, Array<{date: number, stock: number, platform: string}>>} Temizlenmis veri
   */
  function pruneTrackingData(data) {
    const cutoff = Date.now() - TRACKING_RETENTION_DAYS * 24 * 60 * 60 * 1000;

    /** @type {Record<string, Array<{date: number, stock: number, platform: string}>>} */
    const pruned = {};

    for (const [productId, entries] of Object.entries(data)) {
      const filtered = entries.filter(
        (entry) => entry.date >= cutoff
      );
      if (filtered.length > 0) {
        pruned[productId] = filtered;
      }
    }

    // Maksimum urun sinirini asiyorsa en eski urunleri kaldir
    const productIds = Object.keys(pruned);
    if (productIds.length > MAX_TRACKED_PRODUCTS) {
      // Her urunun en son kaydinin tarihine gore sirala, en eskileri at
      const sorted = productIds
        .map((id) => ({
          id,
          lastDate: Math.max(...pruned[id].map((e) => e.date)),
        }))
        .sort((a, b) => b.lastDate - a.lastDate);

      const toRemove = sorted.slice(MAX_TRACKED_PRODUCTS);
      for (const { id } of toRemove) {
        delete pruned[id];
      }
    }

    return pruned;
  }

  // ---------------------------------------------------------------------------
  // Ana API
  // ---------------------------------------------------------------------------

  /**
   * Mevcut sinyallerden aylik satis tahmini uretir.
   *
   * @param {object} signals - Urun sinyalleri (tum alanlar opsiyonel)
   * @returns {object} SalesEstimate sonuc nesnesi
   *
   * @example
   * const estimate = PazarRadar.SalesEstimator.estimate({
   *   platform: 'trendyol',
   *   categoryKey: 'elektronik',
   *   totalReviews: 245,
   *   favoriteCount: 1200,
   *   currentPrice: 299.90,
   *   listingAge: 180,
   * });
   */
  function estimate(signals) {
    if (!signals || typeof signals !== 'object') {
      return _createEmptyEstimate();
    }

    const category = resolveCategory(signals.categoryKey);
    const currentPrice = safeNumber(signals.currentPrice, 0);

    // Farkli sinyal kaynaklarindan tahminler
    const reviewBased = estimateFromReviews(signals, category);
    const favoriteBased = estimateFromFavorites(signals, category);
    const stockBased = estimateFromStock(signals);

    // Agirlikli birlesik tahmin
    const { combined, confidence } = computeWeightedEstimate(
      reviewBased,
      favoriteBased,
      stockBased
    );

    const estimatedMonthlySales = Math.round(combined);
    const estimatedMonthlyRevenue = Math.round(combined * currentPrice);

    // Talep seviyesi
    const demandLevel = determineDemandLevel(estimatedMonthlySales);
    const trend = determineTrend(signals);

    // Rekabet seviyesi
    const competitionLevel = determineCompetitionLevel(
      signals.totalSellersInCategory
    );

    // Gosterim metinleri
    const salesText = formatSalesText(estimatedMonthlySales, confidence);
    const revenueText = formatRevenueText(estimatedMonthlyRevenue);
    const demandText = DEMAND_LABELS[demandLevel] || 'Bilinmiyor';

    return {
      estimatedMonthlySales,
      estimatedMonthlyRevenue,
      confidence,
      signalBreakdown: {
        reviewBased: Math.round(reviewBased),
        favoriteBased: Math.round(favoriteBased),
        stockBased: stockBased != null ? Math.round(stockBased) : null,
        combined: estimatedMonthlySales,
      },
      demandLevel,
      trend,
      competitionLevel,
      salesText,
      revenueText,
      demandText,
    };
  }

  /**
   * Sinyal yokken dondurulecek bos tahmin nesnesi.
   * @returns {object}
   */
  function _createEmptyEstimate() {
    return {
      estimatedMonthlySales: 0,
      estimatedMonthlyRevenue: 0,
      confidence: 'low',
      signalBreakdown: {
        reviewBased: 0,
        favoriteBased: 0,
        stockBased: null,
        combined: 0,
      },
      demandLevel: 'very_low',
      trend: 'unknown',
      competitionLevel: 'medium',
      salesText: '0/ay',
      revenueText: '0 TL/ay',
      demandText: 'Cok Dusuk Talep',
    };
  }

  /**
   * Kategori sinyallerinden nis/kategori skoru hesaplar.
   *
   * @param {object} categorySignals - Kategori geneli sinyaller
   * @param {number} [categorySignals.avgMonthlySales]      - Kategorideki ortalama aylik satis
   * @param {number} [categorySignals.categoryAvgReviews]   - Kategorideki ortalama yorum sayisi
   * @param {number} [categorySignals.categoryAvgPrice]     - Kategorideki ortalama fiyat
   * @param {number} [categorySignals.totalSellersInCategory] - Kategorideki toplam satici sayisi
   * @param {number} [categorySignals.avgMargin]            - Ortalama kar marji (0-1)
   * @param {number} [categorySignals.reviewVelocityTrend]  - Yorum hizi trendi (-1 ile 1 arasi)
   * @param {number} [categorySignals.avgListingQuality]    - Ortalama ilan kalitesi (0-10)
   * @param {number} [categorySignals.avgMonthlySalesBaseline] - Karsilastirma icin baz satis degeri
   * @returns {object} NicheScore sonuc nesnesi
   *
   * @example
   * const niche = PazarRadar.SalesEstimator.nicheScore({
   *   avgMonthlySales: 320,
   *   totalSellersInCategory: 45,
   *   avgMargin: 0.35,
   *   reviewVelocityTrend: 0.5,
   *   avgListingQuality: 4.5,
   * });
   */
  function nicheScore(categorySignals) {
    if (!categorySignals || typeof categorySignals !== 'object') {
      return _createDefaultNicheScore();
    }

    const avgSales = safeNumber(categorySignals.avgMonthlySales, 0);
    const totalSellers = safeNumber(
      categorySignals.totalSellersInCategory,
      50
    );
    const avgMargin = safeNumber(categorySignals.avgMargin, 0.2);
    const velocityTrend = safeNumber(
      categorySignals.reviewVelocityTrend,
      0
    );
    const listingQuality = safeNumber(
      categorySignals.avgListingQuality,
      5
    );
    const salesBaseline = safeNumber(
      categorySignals.avgMonthlySalesBaseline,
      100
    );

    // Talep skoru: ortalama satislarin baz degere orani
    const demandRaw = salesBaseline > 0 ? avgSales / salesBaseline : 0;
    const demand = normalize(demandRaw, 0, 3);

    // Rekabet skoru: az satici = yuksek skor (iyi)
    const competitionRaw = normalize(totalSellers, 0, 200);
    const competition = 11 - competitionRaw; // Ters cevir: 10=az satici=iyi

    // Karlilik skoru
    const profitability = normalize(avgMargin, 0, 0.6);

    // Trend skoru: -1 (dusen) ile 1 (yukselis) arasinda
    const trend = normalize(velocityTrend, -1, 1);

    // Listeleme kalite boslugu: dusuk kalite = yuksek firsat
    const qualityGapRaw = 10 - listingQuality;
    const listingQualityGap = normalize(qualityGapRaw, 0, 10);

    // Genel skor (agirlikli ortalama)
    const overall =
      demand * 0.3 +
      competition * 0.25 +
      profitability * 0.25 +
      trend * 0.1 +
      listingQualityGap * 0.1;

    const roundedOverall = Math.round(overall * 10) / 10;

    // Karar metni
    let verdict;
    if (roundedOverall >= 8) {
      verdict = 'excellent';
    } else if (roundedOverall >= 6.5) {
      verdict = 'good';
    } else if (roundedOverall >= 4.5) {
      verdict = 'moderate';
    } else if (roundedOverall >= 3) {
      verdict = 'risky';
    } else {
      verdict = 'avoid';
    }

    const verdictInfo = VERDICT_MAP[verdict];

    return {
      overall: roundedOverall,
      demand: Math.round(demand * 10) / 10,
      competition: Math.round(competition * 10) / 10,
      profitability: Math.round(profitability * 10) / 10,
      trend: Math.round(trend * 10) / 10,
      listingQualityGap: Math.round(listingQualityGap * 10) / 10,
      verdict,
      verdictText: verdictInfo.text,
      color: verdictInfo.color,
    };
  }

  /**
   * Varsayilan nis skoru nesnesi.
   * @returns {object}
   */
  function _createDefaultNicheScore() {
    return {
      overall: 5,
      demand: 5,
      competition: 5,
      profitability: 5,
      trend: 5,
      listingQualityGap: 5,
      verdict: 'moderate',
      verdictText: VERDICT_MAP.moderate.text,
      color: VERDICT_MAP.moderate.color,
    };
  }

  /**
   * Bir urunu stok takibine alir. Her cagrildiginda mevcut stok degerini
   * tarih damgasiyla birlikte kaydeder.
   *
   * @param {string} productId    - Urun benzersiz kimligi
   * @param {string} platform     - Pazaryeri adi ('trendyol' | 'hepsiburada' | 'n11')
   * @param {number} currentStock - Mevcut stok miktari
   * @returns {Promise<void>}
   *
   * @example
   * await PazarRadar.SalesEstimator.trackProduct(
   *   'TY-12345',
   *   'trendyol',
   *   48
   * );
   */
  async function trackProduct(productId, platform, currentStock) {
    if (!productId || currentStock == null) return;

    const stock = safeNumber(currentStock, 0);
    const data = await loadTrackingData();

    const entry = {
      date: Date.now(),
      stock,
      platform: platform || 'unknown',
    };

    // Mevcut kayitlara ekle veya yeni baslik olustur
    const existing = data[productId] || [];
    const updated = {
      ...data,
      [productId]: [...existing, entry],
    };

    // Eski kayitlari ve fazla urunleri temizle
    const pruned = pruneTrackingData(updated);

    await saveTrackingData(pruned);
  }

  /**
   * Takip edilen bir urunun stok gecmisini getirir.
   *
   * @param {string} productId - Urun benzersiz kimligi
   * @returns {Promise<Array<{date: number, stock: number}>>} Tarih-stok ciftleri
   *
   * @example
   * const history = await PazarRadar.SalesEstimator.getTrackedHistory('TY-12345');
   * // [{ date: 1707350400000, stock: 48 }, { date: 1707436800000, stock: 42 }]
   */
  async function getTrackedHistory(productId) {
    if (!productId) return [];

    const data = await loadTrackingData();
    const entries = data[productId];

    if (!Array.isArray(entries) || entries.length === 0) return [];

    // Sadece date ve stock dondur, tarihe gore sirala
    return entries
      .map((entry) => ({
        date: entry.date,
        stock: entry.stock,
      }))
      .sort((a, b) => a.date - b.date);
  }

  // ---------------------------------------------------------------------------
  // Namespace'e ata
  // ---------------------------------------------------------------------------

  window.PazarRadar.SalesEstimator = {
    estimate,
    nicheScore,
    trackProduct,
    getTrackedHistory,
  };
})();
