/**
 * PazarRadar - Popup Script
 *
 * Hizli kar hesaplama, kullanim istatistikleri, son analizler
 * ve plan bilgisi gosteren popup arayuzu.
 *
 * background.js ile chrome.runtime.sendMessage araciligiyla iletisim kurar.
 */
(function () {
  'use strict';

  // ─── Sabitler ───────────────────────────────────────────────

  /** KDV orani */
  const DEFAULT_KDV_RATE = 20;

  /** Hizmet KDV orani (komisyon ve platform ucreti uzerinden) */
  const SERVICE_KDV_RATE = 0.20;

  /** Platform yapilandirmalari (profit-calculator.js ile senkron) */
  const PLATFORM_CONFIG = Object.freeze({
    trendyol: {
      name: 'Trendyol',
      commissionBase: 'kdv_haric',
      platformFee: 8.49,
      defaultCargoPerDesi: 14.99,
      minCargo: 14.99
    },
    hepsiburada: {
      name: 'Hepsiburada',
      commissionBase: 'kdv_dahil',
      platformFee: 10.00,
      defaultCargoPerDesi: 16.49,
      minCargo: 16.49
    },
    n11: {
      name: 'N11',
      commissionBase: 'kdv_haric',
      platformFee: 7.00,
      defaultCargoPerDesi: 13.99,
      minCargo: 13.99
    }
  });

  /** Gecerli platformlar */
  const VALID_PLATFORMS = ['trendyol', 'hepsiburada', 'n11'];

  /**
   * Kategori komisyon oranlari - CommissionDB'den dinamik yuklenir.
   * Tek kaynak ilkesi: lib/commission-db.js ana veritabani.
   */
  const COMMISSION_RATES = (function buildRatesFromDB() {
    const PR = window.PazarRadar;
    const db = PR && PR.CommissionDB;
    if (!db || typeof db.getCategories !== 'function') {
      return { trendyol: {}, hepsiburada: {}, n11: {} };
    }
    const rates = {};
    ['trendyol', 'hepsiburada', 'n11'].forEach(function (platform) {
      rates[platform] = {};
      const cats = db.getCategories(platform);
      if (cats) {
        cats.forEach(function (cat) {
          rates[platform][cat.key] = { name: cat.name, rate: cat.rate };
        });
      }
    });
    return Object.freeze(rates);
  })();

  /** Kategori ikonlari */
  const CATEGORY_ICONS = Object.freeze({
    elektronik: '\uD83D\uDCF1',
    giyim: '\uD83D\uDC55',
    ayakkabi: '\uD83D\uDC5F',
    canta: '\uD83D\uDC5C',
    kozmetik: '\uD83D\uDC84',
    ev: '\uD83C\uDFE0',
    mobilya: '\uD83D\uDECB\uFE0F',
    spor: '\u26BD',
    oyuncak: '\uD83E\uDDF8',
    kitap: '\uD83D\uDCDA',
    anne: '\uD83D\uDC76',
    pet: '\uD83D\uDC3E',
    oto: '\uD83D\uDE97',
    gida: '\uD83C\uDF4E',
    saat: '\u231A',
    supermarket: '\uD83D\uDED2',
    mutfak: '\uD83C\uDF73',
    dekorasyon: '\uD83C\uDFA8',
    default: '\uD83D\uDCE6'
  });

  /** Plan yapilandirmalari */
  const PLAN_CONFIG = Object.freeze({
    free: { label: 'Ucretsiz Plan', icon: '\uD83C\uDD93', cssClass: 'pr-plan-free' },
    trial: { label: 'PRO Deneme', icon: '\u2728', cssClass: 'pr-plan-trial' },
    pro: { label: 'PRO Plan', icon: '\uD83D\uDE80', cssClass: 'pr-plan-pro' },
    business: { label: 'Business Plan', icon: '\uD83C\uDFC6', cssClass: 'pr-plan-business' }
  });

  // ─── Yardimci Fonksiyonlar ──────────────────────────────────

  /**
   * Sayiyi 2 ondalik basamaga yuvarlar.
   * @param {number} value
   * @returns {number}
   */
  function round2(value) {
    return Math.round((value + Number.EPSILON) * 100) / 100;
  }

  /**
   * Sayiyi Turkce formatta gosterir (1.234,56 TL).
   * @param {number} value
   * @param {boolean} [showCurrency=true]
   * @returns {string}
   */
  function formatTL(value, showCurrency) {
    if (value == null || Number.isNaN(value)) {
      return showCurrency !== false ? '-- TL' : '--';
    }

    const formatted = new Intl.NumberFormat('tr-TR', {
      minimumFractionDigits: 2,
      maximumFractionDigits: 2
    }).format(value);

    return showCurrency !== false ? formatted + ' TL' : formatted;
  }

  /**
   * Yuzde degerini formatlar.
   * @param {number} value
   * @returns {string}
   */
  function formatPercent(value) {
    if (value == null || Number.isNaN(value)) return '--%';
    return '%' + new Intl.NumberFormat('tr-TR', {
      minimumFractionDigits: 1,
      maximumFractionDigits: 1
    }).format(value);
  }

  /**
   * Guvenli DOM element olusturucu.
   * @param {string} tag
   * @param {Object} [attrs]
   * @param {Array<Node|string>} [children]
   * @returns {HTMLElement}
   */
  function el(tag, attrs, children) {
    const element = document.createElement(tag);

    if (attrs) {
      for (const [key, value] of Object.entries(attrs)) {
        if (key === 'className') {
          element.className = value;
        } else if (key === 'textContent') {
          element.textContent = value;
        } else if (key === 'dataset') {
          for (const [dk, dv] of Object.entries(value)) {
            element.dataset[dk] = dv;
          }
        } else if (key.startsWith('on') && typeof value === 'function') {
          element.addEventListener(key.slice(2).toLowerCase(), value);
        } else {
          element.setAttribute(key, value);
        }
      }
    }

    if (children) {
      for (const child of children) {
        if (typeof child === 'string') {
          element.appendChild(document.createTextNode(child));
        } else if (child instanceof Node) {
          element.appendChild(child);
        }
      }
    }

    return element;
  }

  /**
   * Bir container icindeki tum child node'lari temizler.
   * @param {HTMLElement} container
   */
  function clearChildren(container) {
    while (container.firstChild) {
      container.removeChild(container.firstChild);
    }
  }

  /**
   * Kategori anahtarindan ikon secer.
   * @param {string} categoryKey
   * @returns {string}
   */
  function getCategoryIcon(categoryKey) {
    if (!categoryKey) return CATEGORY_ICONS.default;

    const key = categoryKey.toLowerCase();
    for (const [pattern, icon] of Object.entries(CATEGORY_ICONS)) {
      if (key.includes(pattern)) return icon;
    }
    return CATEGORY_ICONS.default;
  }

  /**
   * background.js ile mesaj gonderip yanit bekler.
   * @param {Object} message
   * @returns {Promise<Object>}
   */
  function sendMessage(message) {
    return new Promise(function (resolve) {
      try {
        chrome.runtime.sendMessage(message, function (response) {
          if (chrome.runtime.lastError) {
            resolve({ success: false, error: chrome.runtime.lastError.message });
            return;
          }
          resolve(response || { success: false, error: 'Yanit alinamadi' });
        });
      } catch (err) {
        resolve({ success: false, error: err.message });
      }
    });
  }

  // ─── Yerel Kar Hesaplama Motoru ─────────────────────────────

  /**
   * Tek platform icin kar hesaplamasi (popup icin yerel kopya).
   * @param {string} platform
   * @param {string} categoryKey
   * @param {number} salePrice
   * @param {number|null} costPrice
   * @param {number} cargoWeight - Desi agirlik
   * @returns {Object}
   */
  function calculateProfit(platform, categoryKey, salePrice, costPrice, cargoWeight) {
    const config = PLATFORM_CONFIG[platform];
    if (!config) return null;

    const desi = cargoWeight || 1;

    // Komisyon oranini bul
    const platformRates = COMMISSION_RATES[platform];
    const categoryData = platformRates ? platformRates[categoryKey] : null;
    const commissionRate = categoryData ? categoryData.rate : 0;

    // KDV haric fiyat
    const kdvMultiplier = 1 + (DEFAULT_KDV_RATE / 100);
    const salePriceExKDV = round2(salePrice / kdvMultiplier);

    // Komisyon hesaplama - platforma gore matrah
    let commissionAmount;
    if (config.commissionBase === 'kdv_dahil') {
      commissionAmount = round2(salePrice * commissionRate / 100);
    } else {
      commissionAmount = round2(salePriceExKDV * commissionRate / 100);
    }

    // Komisyon KDV
    const commissionKDV = round2(commissionAmount * SERVICE_KDV_RATE);

    // Platform hizmet bedeli
    const platformFee = config.platformFee;
    const platformFeeKDV = round2(platformFee * SERVICE_KDV_RATE);

    // Kargo (desi bazli)
    const estimatedCargo = round2(Math.max(config.minCargo, config.defaultCargoPerDesi * desi));

    // Toplam kesintiler
    const totalCosts = round2(
      commissionAmount + commissionKDV + platformFee + platformFeeKDV + estimatedCargo
    );

    // Net gelir
    const netRevenue = round2(salePrice - totalCosts);

    // Net kar
    const netProfit = (costPrice != null) ? round2(netRevenue - costPrice) : null;

    // Kar marji
    let netMarginPercent = null;
    if (netProfit != null && salePrice > 0) {
      netMarginPercent = round2((netProfit / salePrice) * 100);
    }

    return {
      platform: platform,
      platformName: config.name,
      salePrice: round2(salePrice),
      commissionRate: commissionRate,
      commissionAmount: commissionAmount,
      commissionKDV: commissionKDV,
      platformFee: round2(platformFee),
      platformFeeKDV: platformFeeKDV,
      estimatedCargo: estimatedCargo,
      totalCosts: totalCosts,
      netRevenue: netRevenue,
      netProfit: netProfit,
      netMarginPercent: netMarginPercent
    };
  }

  /**
   * 3 platform icin karsilastirmali hesaplama yapar.
   * @param {string} categoryKey
   * @param {number} salePrice
   * @param {number|null} costPrice
   * @param {number} cargoWeight
   * @returns {Object}
   */
  function compareAllPlatforms(categoryKey, salePrice, costPrice, cargoWeight) {
    const results = {};
    let bestPlatform = '';
    let bestNetRevenue = -Infinity;

    for (const platform of VALID_PLATFORMS) {
      const result = calculateProfit(platform, categoryKey, salePrice, costPrice, cargoWeight);
      results[platform] = result;

      if (result && result.netRevenue > bestNetRevenue) {
        bestNetRevenue = result.netRevenue;
        bestPlatform = platform;
      }
    }

    return {
      trendyol: results.trendyol || null,
      hepsiburada: results.hepsiburada || null,
      n11: results.n11 || null,
      best: bestPlatform
    };
  }

  // ─── DOM Referanslari ───────────────────────────────────────

  /** @type {HTMLSelectElement} */
  let platformSelect;
  /** @type {HTMLSelectElement} */
  let categorySelect;
  /** @type {HTMLInputElement} */
  let salePriceInput;
  /** @type {HTMLInputElement} */
  let costPriceInput;
  /** @type {HTMLInputElement} */
  let cargoDesiInput;
  /** @type {HTMLElement} */
  let resultCard;
  /** @type {HTMLElement} */
  let resultHero;
  /** @type {HTMLElement} */
  let netProfitEl;
  /** @type {HTMLElement} */
  let marginEl;
  /** @type {HTMLElement} */
  let breakdownBody;
  /** @type {HTMLElement} */
  let totalCostEl;
  /** @type {HTMLElement} */
  let compareBarsEl;
  /** @type {HTMLElement} */
  let statsContent;
  /** @type {HTMLElement} */
  let recentContent;
  /** @type {HTMLElement} */
  let planContent;

  // ─── Kategori Dropdown Doldurma ─────────────────────────────

  /**
   * Secili platforma gore kategori dropdown'ini doldurur.
   */
  function populateCategories() {
    const platform = platformSelect.value;
    const rates = COMMISSION_RATES[platform];

    // Mevcut secimi hatirla
    const currentValue = categorySelect.value;

    // Temizle
    clearChildren(categorySelect);

    // Varsayilan secenek
    const defaultOption = el('option', { value: '', textContent: 'Kategori Secin' });
    categorySelect.appendChild(defaultOption);

    if (!rates) return;

    // Kategorileri ekle
    for (const [key, data] of Object.entries(rates)) {
      const icon = getCategoryIcon(key);
      const optionText = icon + ' ' + data.name + ' (%' + data.rate + ')';
      const option = el('option', { value: key, textContent: optionText });
      categorySelect.appendChild(option);
    }

    // Onceki secimi geri yukle (eger hala mevcutsa)
    if (currentValue && rates[currentValue]) {
      categorySelect.value = currentValue;
    }
  }

  // ─── Hesaplama ve Sonuc Gosterme ───────────────────────────

  /**
   * Girdi degerlerine gore kar hesaplar ve sonuclari gosterir.
   */
  function performCalculation() {
    const salePrice = parseFloat(salePriceInput.value);
    const costPrice = costPriceInput.value ? parseFloat(costPriceInput.value) : null;
    const categoryKey = categorySelect.value;
    const cargoWeight = parseInt(cargoDesiInput.value, 10) || 1;

    // En az satis fiyati girilmeli
    if (!salePrice || salePrice <= 0 || !categoryKey) {
      resultCard.classList.add('pr-hidden');
      return;
    }

    // Karsilastirmali hesaplama
    const comparison = compareAllPlatforms(categoryKey, salePrice, costPrice, cargoWeight);
    const selectedPlatform = platformSelect.value;
    const selectedResult = comparison[selectedPlatform];

    if (!selectedResult) {
      resultCard.classList.add('pr-hidden');
      return;
    }

    // Sonuc kartini goster
    resultCard.classList.remove('pr-hidden');

    // ── Hero sonuc ──
    const displayValue = selectedResult.netProfit != null ? selectedResult.netProfit : selectedResult.netRevenue;
    const heroLabel = selectedResult.netProfit != null ? 'NET KAR' : 'NET GELIR';
    resultHero.querySelector('.pr-result__hero-label').textContent = heroLabel;

    netProfitEl.textContent = formatTL(displayValue);
    if (displayValue > 0) {
      resultHero.className = 'pr-result__hero pr-result__hero--profit';
    } else if (displayValue < 0) {
      resultHero.className = 'pr-result__hero pr-result__hero--loss';
    } else {
      resultHero.className = 'pr-result__hero pr-result__hero--neutral';
    }

    // Marj
    if (selectedResult.netMarginPercent != null) {
      marginEl.textContent = formatPercent(selectedResult.netMarginPercent) + ' marj';
    } else {
      marginEl.textContent = 'Maliyet girin \u2192 marj gorsun';
    }

    // ── Maliyet dokumu tablosu ──
    renderBreakdownTable(selectedResult);

    // ── Platform karsilastirma bar chart ──
    renderComparisonBars(comparison);
  }

  /**
   * Maliyet dokumu tablosunu doldurur.
   * @param {Object} r - Hesaplama sonucu
   */
  function renderBreakdownTable(r) {
    clearChildren(breakdownBody);

    const rows = [
      { label: 'Komisyon (%' + r.commissionRate + ')', value: r.commissionAmount, cls: '' },
      { label: 'Komisyon KDV', value: r.commissionKDV, cls: 'pr-muted' },
      { label: 'Platform Ucreti', value: r.platformFee, cls: '' },
      { label: 'Platform KDV', value: r.platformFeeKDV, cls: 'pr-muted' },
      { label: 'Kargo', value: r.estimatedCargo, cls: '' }
    ];

    for (const item of rows) {
      const tr = document.createElement('tr');
      if (item.cls) tr.className = item.cls;
      const tdLabel = document.createElement('td');
      tdLabel.textContent = item.label;
      const tdValue = document.createElement('td');
      tdValue.textContent = formatTL(item.value);
      tr.appendChild(tdLabel);
      tr.appendChild(tdValue);
      breakdownBody.appendChild(tr);
    }

    totalCostEl.textContent = formatTL(r.totalCosts);
  }

  /**
   * Platform karsilastirma bar chart render eder.
   * @param {Object} comparison
   */
  function renderComparisonBars(comparison) {
    clearChildren(compareBarsEl);

    // Maksimum degeri bul (bar genisligi icin)
    let maxVal = 0;
    for (const p of VALID_PLATFORMS) {
      const r = comparison[p];
      if (r) {
        const v = r.netProfit != null ? r.netProfit : r.netRevenue;
        if (Math.abs(v) > maxVal) maxVal = Math.abs(v);
      }
    }
    if (maxVal === 0) maxVal = 1;

    const PLATFORM_COLORS = {
      trendyol: '#f97316',
      hepsiburada: '#ff6600',
      n11: '#9b59b6'
    };

    for (const p of VALID_PLATFORMS) {
      const r = comparison[p];
      if (!r) continue;

      const isBest = (p === comparison.best);
      const v = r.netProfit != null ? r.netProfit : r.netRevenue;
      const pct = Math.max(5, Math.round((Math.abs(v) / maxVal) * 100));

      const row = el('div', { className: 'pr-compare-bar-row' + (isBest ? ' pr-compare-bar-row--best' : '') });

      // Platform adi
      const label = el('div', { className: 'pr-compare-bar-label' });
      label.textContent = r.platformName;
      if (isBest) {
        const star = el('span', { className: 'pr-compare-bar-star', textContent: ' EN IYI' });
        label.appendChild(star);
      }

      // Bar
      const barWrap = el('div', { className: 'pr-compare-bar-wrap' });
      const bar = el('div', { className: 'pr-compare-bar' });
      bar.style.width = pct + '%';
      bar.style.background = PLATFORM_COLORS[p] || '#6366f1';
      if (v < 0) bar.classList.add('pr-compare-bar--negative');
      barWrap.appendChild(bar);

      // Deger
      const valueEl = el('div', { className: 'pr-compare-bar-value' + (v < 0 ? ' pr-negative' : ' pr-positive') });
      valueEl.textContent = formatTL(v);
      if (r.netMarginPercent != null) {
        valueEl.textContent += ' (%' + r.netMarginPercent + ')';
      }

      row.appendChild(label);
      row.appendChild(barWrap);
      row.appendChild(valueEl);
      compareBarsEl.appendChild(row);
    }
  }

  // ─── Istatistikleri Yukleme ─────────────────────────────────

  /**
   * Bugunun kullanim istatistiklerini background'dan alip render eder.
   */
  async function loadUsageStats() {
    const response = await sendMessage({ action: 'get_usage_stats' });

    clearChildren(statsContent);

    if (!response || !response.success) {
      // Varsayilan veriler goster
      renderStats({ analysisUsed: 0, analysisLimit: 5, pagesUsed: 0, pagesLimit: 3 });
      return;
    }

    renderStats(response.data || { analysisUsed: 0, analysisLimit: 5, pagesUsed: 0, pagesLimit: 3 });
  }

  /**
   * Istatistik verisini DOM'a render eder.
   * @param {Object} stats
   */
  function renderStats(stats) {
    const analysisUsed = stats.analysisUsed || 0;
    const analysisLimit = stats.analysisLimit || 5;
    const pagesUsed = stats.pagesUsed || 0;
    const pagesLimit = stats.pagesLimit || 3;

    // Istatistik satirlari
    const statsRow = el('div', { className: 'pr-stats-row' });

    // Analiz sayisi
    const analysisStat = el('span', { className: 'pr-stat-item' }, [
      el('span', { className: 'pr-stat-icon', textContent: '\uD83D\uDCCA' }),
      el('span', { className: 'pr-stat-text', textContent: analysisUsed + '/' + analysisLimit + ' analiz' })
    ]);
    statsRow.appendChild(analysisStat);

    // Sayfa sayisi
    const pagesStat = el('span', { className: 'pr-stat-item' }, [
      el('span', { className: 'pr-stat-icon', textContent: '\uD83D\uDCC4' }),
      el('span', { className: 'pr-stat-text', textContent: pagesUsed + '/' + pagesLimit + ' sayfa' })
    ]);
    statsRow.appendChild(pagesStat);

    statsContent.appendChild(statsRow);

    // Progress bar
    const progressPercent = analysisLimit > 0 ? Math.min((analysisUsed / analysisLimit) * 100, 100) : 0;
    let progressClass = 'pr-progress-fill';
    if (progressPercent >= 100) {
      progressClass += ' pr-full';
    } else if (progressPercent >= 70) {
      progressClass += ' pr-warn';
    }

    const progressBar = el('div', { className: 'pr-progress-bar' }, [
      el('div', { className: progressClass, style: 'width: ' + progressPercent + '%' })
    ]);
    statsContent.appendChild(progressBar);
  }

  // ─── Son Analizler ──────────────────────────────────────────

  /**
   * Son analizleri background'dan alip render eder.
   */
  async function loadRecentAnalyses() {
    const response = await sendMessage({ action: 'get_recent_analyses' });

    clearChildren(recentContent);

    if (!response || !response.success || !response.data || response.data.length === 0) {
      renderEmptyState(recentContent, 'Henuz analiz yapilmadi');
      return;
    }

    renderRecentAnalyses(response.data);
  }

  /**
   * Son analiz listesini render eder.
   * @param {Array<Object>} analyses
   */
  function renderRecentAnalyses(analyses) {
    const list = el('div', { className: 'pr-analysis-list pr-fade-in' });

    // Son 3 analizi goster
    const displayList = analyses.slice(0, 3);

    for (const item of displayList) {
      const icon = getCategoryIcon(item.categoryKey || item.category || '');
      const marginValue = item.netMarginPercent || item.margin || 0;
      const platformName = item.platformName || PLATFORM_CONFIG[item.platform]?.name || item.platform || '';

      const row = el('div', { className: 'pr-analysis-item' });

      row.appendChild(el('span', { className: 'pr-analysis-icon', textContent: icon }));

      const info = el('div', { className: 'pr-analysis-info' });
      info.appendChild(el('div', { className: 'pr-analysis-name', textContent: item.productName || item.name || 'Urun' }));
      info.appendChild(el('div', { className: 'pr-analysis-platform', textContent: platformName }));
      row.appendChild(info);

      const marginClass = marginValue >= 0 ? 'pr-positive' : 'pr-negative';
      row.appendChild(el('span', {
        className: 'pr-analysis-margin ' + marginClass,
        textContent: formatPercent(marginValue)
      }));

      list.appendChild(row);
    }

    recentContent.appendChild(list);
  }

  /**
   * Bos durum mesaji gosterir.
   * @param {HTMLElement} container
   * @param {string} message
   */
  function renderEmptyState(container, message) {
    const emptyEl = el('div', { className: 'pr-empty-state', textContent: message });
    container.appendChild(emptyEl);
  }

  // ─── Plan Bilgisi ───────────────────────────────────────────

  /**
   * Plan bilgisini ve kullanim limitini background'dan alip render eder.
   */
  async function loadPlanInfo() {
    const [settingsResponse, usageResponse] = await Promise.all([
      sendMessage({ type: 'get_settings' }),
      sendMessage({ type: 'peek_usage', payload: { usageType: 'productAnalyses' } })
    ]);

    clearChildren(planContent);

    const plan = (settingsResponse && settingsResponse.success && settingsResponse.data)
      ? (settingsResponse.data.plan || 'free')
      : 'free';

    const trialEndsAt = (settingsResponse && settingsResponse.success && settingsResponse.data)
      ? (settingsResponse.data.trialEndsAt || null)
      : null;

    const gracePeriod = (settingsResponse && settingsResponse.success && settingsResponse.data)
      ? (settingsResponse.data.gracePeriod || null)
      : null;

    const usageInfo = (usageResponse && usageResponse.success)
      ? usageResponse
      : { current: 0, limit: 5, remaining: 5 };

    renderPlanSection(plan, usageInfo, trialEndsAt, gracePeriod);
  }

  /**
   * Plan bolumunu render eder - lisans girisi ve kalan hak gostergesi dahil.
   * @param {string} plan - 'free' | 'trial' | 'pro' | 'business'
   * @param {Object} usageInfo - {current, limit, remaining}
   * @param {number|null} trialEndsAt - Trial bitis timestamp'i
   */
  function renderPlanSection(plan, usageInfo, trialEndsAt, gracePeriod) {
    const planConfig = PLAN_CONFIG[plan] || PLAN_CONFIG.free;

    // Plan badge satiri
    const planRow = el('div', { className: 'pr-plan-row' });

    const badge = el('span', {
      className: 'pr-plan-badge ' + planConfig.cssClass,
      textContent: planConfig.icon + ' ' + planConfig.label
    });
    planRow.appendChild(badge);

    // Kalan hak gostergesi
    if (plan === 'trial' && trialEndsAt) {
      const msLeft = trialEndsAt - Date.now();
      const daysLeft = Math.max(0, Math.ceil(msLeft / (24 * 60 * 60 * 1000)));
      const trialInfo = el('span', {
        className: 'pr-plan-remaining pr-trial-text',
        textContent: daysLeft + ' g\u00fcn kald\u0131'
      });
      planRow.appendChild(trialInfo);
    } else if (plan === 'free' && usageInfo) {
      const remaining = el('span', {
        className: 'pr-plan-remaining',
        textContent: 'Kalan: ' + (usageInfo.remaining || 0) + '/' + (usageInfo.limit || 5) + ' analiz'
      });
      planRow.appendChild(remaining);
    } else if (plan !== 'free') {
      const unlimited = el('span', {
        className: 'pr-plan-remaining pr-positive',
        textContent: '\u2713 S\u0131n\u0131rs\u0131z'
      });
      planRow.appendChild(unlimited);
    }

    planContent.appendChild(planRow);

    // Grace period uyarisi (sunucuya erisilemiyorsa)
    if (gracePeriod && gracePeriod.active) {
      const graceRow = el('div', { className: 'pr-grace-warning' });
      graceRow.textContent = '\u26A0 Sunucu dogrulamasi basarisiz. Kalan sure: ' + gracePeriod.daysLeft + ' gun';
      planContent.appendChild(graceRow);
    }

    // Lisans girisi (sadece free plan icin)
    if (plan === 'free') {
      const licenseRow = el('div', { className: 'pr-license-row' });

      const licenseInput = el('input', {
        className: 'pr-license-input',
        type: 'text',
        placeholder: 'Lisans Anahtari (PR-XXXX-...)'
      });
      licenseRow.appendChild(licenseInput);

      const activateBtn = el('button', {
        className: 'pr-license-btn',
        textContent: 'Aktive Et'
      });

      const resultMsg = el('div', { className: 'pr-license-result' });

      activateBtn.addEventListener('click', async function () {
        const key = licenseInput.value.trim();
        if (!key) {
          resultMsg.textContent = 'Lisans anahtarini girin';
          resultMsg.className = 'pr-license-result pr-negative';
          return;
        }

        activateBtn.textContent = 'Kontrol ediliyor...';
        activateBtn.disabled = true;

        const response = await sendMessage({
          type: 'activate_license',
          payload: { key: key }
        });

        if (response && response.success) {
          resultMsg.textContent = '\u2705 PRO aktif!';
          resultMsg.className = 'pr-license-result pr-positive';
          // Popup'i yeniden yukle
          setTimeout(function () { location.reload(); }, 1000);
        } else {
          resultMsg.textContent = (response && response.error) || 'Gecersiz anahtar';
          resultMsg.className = 'pr-license-result pr-negative';
          activateBtn.textContent = 'Aktive Et';
          activateBtn.disabled = false;
        }
      });

      licenseRow.appendChild(activateBtn);
      planContent.appendChild(licenseRow);
      planContent.appendChild(resultMsg);

      // Yukseltme linki
      const upgradeBtn = el('button', {
        className: 'pr-upgrade-btn',
        textContent: "\uD83D\uDE80 PRO'ya Yukselt",
        onClick: handleUpgradeClick
      });
      planContent.appendChild(upgradeBtn);
    }
  }

  // ─── Event Handler'lar ──────────────────────────────────────

  /**
   * Platform degistiginde kategorileri gunceller ve yeniden hesaplar.
   */
  function handlePlatformChange() {
    populateCategories();
    performCalculation();
  }

  /**
   * Kategori degistiginde yeniden hesaplar.
   */
  function handleCategoryChange() {
    performCalculation();
  }

  /**
   * Fiyat girisleri degistiginde yeniden hesaplar.
   */
  function handlePriceInput() {
    performCalculation();
  }

  /**
   * Ayarlar butonuna tiklandiginda side panel acar veya options sayfasina yonlendirir.
   */
  function handleSettingsClick() {
    // Side panel acmayi dene
    if (chrome.sidePanel && typeof chrome.sidePanel.open === 'function') {
      chrome.tabs.query({ active: true, currentWindow: true }, function (tabs) {
        if (tabs && tabs[0]) {
          chrome.sidePanel.open({ tabId: tabs[0].id });
        }
      });
      window.close();
      return;
    }

    // Fallback: options sayfasini ac
    if (chrome.runtime && chrome.runtime.openOptionsPage) {
      chrome.runtime.openOptionsPage();
    }
  }

  /**
   * PRO yukseltme butonuna tiklandiginda yonlendirir.
   */
  function handleUpgradeClick() {
    chrome.tabs.create({ url: 'https://pazarradar.teknoweb.net/#fiyatlandirma' });
    window.close();
  }

  // ─── Baslatma ───────────────────────────────────────────────

  /**
   * Popup baslatma fonksiyonu. DOM referanslarini alir, event listener'lari ekler
   * ve verileri yukler.
   */
  function init() {
    // DOM referanslari
    platformSelect = document.getElementById('prPlatform');
    categorySelect = document.getElementById('prCategory');
    salePriceInput = document.getElementById('prSalePrice');
    costPriceInput = document.getElementById('prCostPrice');
    cargoDesiInput = document.getElementById('prCargoDesi');
    resultCard = document.getElementById('prResultCard');
    resultHero = document.getElementById('prResultHero');
    netProfitEl = document.getElementById('prNetProfit');
    marginEl = document.getElementById('prMargin');
    breakdownBody = document.getElementById('prBreakdownBody');
    totalCostEl = document.getElementById('prTotalCost');
    compareBarsEl = document.getElementById('prCompareBars');
    statsContent = document.getElementById('prStatsContent');
    recentContent = document.getElementById('prRecentContent');
    planContent = document.getElementById('prPlanContent');

    // Event listener'lar
    platformSelect.addEventListener('change', handlePlatformChange);
    categorySelect.addEventListener('change', handleCategoryChange);
    salePriceInput.addEventListener('input', handlePriceInput);
    costPriceInput.addEventListener('input', handlePriceInput);
    cargoDesiInput.addEventListener('input', handlePriceInput);

    const settingsBtn = document.getElementById('prSettingsBtn');
    if (settingsBtn) {
      settingsBtn.addEventListener('click', handleSettingsClick);
    }

    // Kategorileri doldur
    populateCategories();

    // Asenkron verileri yukle
    loadUsageStats();
    loadRecentAnalyses();
    loadPlanInfo();
  }

  // DOM hazir olunca baslat
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }
})();
