/**
 * PazarRadar - Side Panel Script
 *
 * Tam urun analizi, son analizler, fiyat alarmlari,
 * ayarlar, lisans yonetimi ve plan bilgisi gosteren panel arayuzu.
 *
 * Content script'lerden gelen urun verisini alir ve render eder.
 * background.js ile chrome.runtime.sendMessage araciligiyla iletisim kurar.
 */
(function () {
  'use strict';

  // ─── Sabitler ───────────────────────────────────────────────

  /** KDV orani */
  const DEFAULT_KDV_RATE = 20;

  /** Hizmet KDV orani */
  const SERVICE_KDV_RATE = 0.20;

  /** Platform yapilandirmalari */
  const PLATFORM_CONFIG = Object.freeze({
    trendyol: {
      name: 'Trendyol',
      commissionBase: 'kdv_haric',
      platformFee: 8.49,
      defaultCargoPerDesi: 14.99,
      minCargo: 14.99
    },
    hepsiburada: {
      name: 'Hepsiburada',
      commissionBase: 'kdv_dahil',
      platformFee: 10.00,
      defaultCargoPerDesi: 16.49,
      minCargo: 16.49
    },
    n11: {
      name: 'N11',
      commissionBase: 'kdv_haric',
      platformFee: 7.00,
      defaultCargoPerDesi: 13.99,
      minCargo: 13.99
    }
  });

  /** Gecerli platformlar */
  const VALID_PLATFORMS = ['trendyol', 'hepsiburada', 'n11'];

  /** Komisyon oranlari (popup.js ile senkron) */
  const COMMISSION_RATES = Object.freeze({
    trendyol: {
      elektronik_telefon: { name: 'Elektronik (Telefon)', rate: 8.5 },
      elektronik_bilgisayar: { name: 'Elektronik (Bilgisayar)', rate: 7.5 },
      elektronik_tv: { name: 'Elektronik (TV & Ses)', rate: 9.0 },
      elektronik_aksesuar: { name: 'Elektronik Aksesuar', rate: 12.0 },
      elektronik_beyaz_esya: { name: 'Beyaz Esya', rate: 6.0 },
      elektronik_kucuk_ev_aletleri: { name: 'Kucuk Ev Aletleri', rate: 10.0 },
      giyim_kadin: { name: 'Kadin Giyim', rate: 18.0 },
      giyim_erkek: { name: 'Erkek Giyim', rate: 18.0 },
      giyim_cocuk: { name: 'Cocuk Giyim', rate: 18.0 },
      ayakkabi: { name: 'Ayakkabi', rate: 18.0 },
      canta_aksesuar: { name: 'Canta & Aksesuar', rate: 18.0 },
      kozmetik: { name: 'Kozmetik & Kisisel Bakim', rate: 14.0 },
      ev_yasam: { name: 'Ev & Yasam', rate: 15.0 },
      ev_mobilya: { name: 'Mobilya', rate: 12.0 },
      ev_dekorasyon: { name: 'Dekorasyon', rate: 15.0 },
      mutfak: { name: 'Mutfak Gerecleri', rate: 15.0 },
      spor_outdoor: { name: 'Spor & Outdoor', rate: 14.0 },
      oyuncak: { name: 'Oyuncak', rate: 15.0 },
      kitap: { name: 'Kitap & Kirtasiye', rate: 16.0 },
      anne_bebek: { name: 'Anne & Bebek', rate: 14.0 },
      pet_urunleri: { name: 'Pet Urunleri', rate: 14.0 },
      oto_aksesuar: { name: 'Oto Aksesuar', rate: 14.0 },
      gida: { name: 'Gida & Icecek', rate: 12.0 },
      saat_takı: { name: 'Saat & Taki', rate: 16.0 },
      supermarket: { name: 'Supermarket', rate: 10.0 }
    },
    hepsiburada: {
      elektronik_telefon: { name: 'Elektronik (Telefon)', rate: 7.0 },
      elektronik_bilgisayar: { name: 'Elektronik (Bilgisayar)', rate: 6.5 },
      elektronik_tv: { name: 'Elektronik (TV & Ses)', rate: 8.0 },
      elektronik_aksesuar: { name: 'Elektronik Aksesuar', rate: 11.0 },
      elektronik_beyaz_esya: { name: 'Beyaz Esya', rate: 5.5 },
      elektronik_kucuk_ev_aletleri: { name: 'Kucuk Ev Aletleri', rate: 9.0 },
      giyim_kadin: { name: 'Kadin Giyim', rate: 19.0 },
      giyim_erkek: { name: 'Erkek Giyim', rate: 19.0 },
      giyim_cocuk: { name: 'Cocuk Giyim', rate: 19.0 },
      ayakkabi: { name: 'Ayakkabi', rate: 19.0 },
      canta_aksesuar: { name: 'Canta & Aksesuar', rate: 19.0 },
      kozmetik: { name: 'Kozmetik & Kisisel Bakim', rate: 13.0 },
      ev_yasam: { name: 'Ev & Yasam', rate: 14.0 },
      ev_mobilya: { name: 'Mobilya', rate: 11.0 },
      ev_dekorasyon: { name: 'Dekorasyon', rate: 14.0 },
      mutfak: { name: 'Mutfak Gerecleri', rate: 14.0 },
      spor_outdoor: { name: 'Spor & Outdoor', rate: 13.0 },
      oyuncak: { name: 'Oyuncak', rate: 14.0 },
      kitap: { name: 'Kitap & Kirtasiye', rate: 15.0 },
      anne_bebek: { name: 'Anne & Bebek', rate: 13.0 },
      pet_urunleri: { name: 'Pet Urunleri', rate: 13.0 },
      oto_aksesuar: { name: 'Oto Aksesuar', rate: 13.0 },
      gida: { name: 'Gida & Icecek', rate: 11.0 },
      saat_takı: { name: 'Saat & Taki', rate: 15.0 },
      supermarket: { name: 'Supermarket', rate: 9.0 }
    },
    n11: {
      elektronik_telefon: { name: 'Elektronik (Telefon)', rate: 7.5 },
      elektronik_bilgisayar: { name: 'Elektronik (Bilgisayar)', rate: 7.0 },
      elektronik_tv: { name: 'Elektronik (TV & Ses)', rate: 8.5 },
      elektronik_aksesuar: { name: 'Elektronik Aksesuar', rate: 11.5 },
      elektronik_beyaz_esya: { name: 'Beyaz Esya', rate: 5.5 },
      elektronik_kucuk_ev_aletleri: { name: 'Kucuk Ev Aletleri', rate: 9.5 },
      giyim_kadin: { name: 'Kadin Giyim', rate: 17.0 },
      giyim_erkek: { name: 'Erkek Giyim', rate: 17.0 },
      giyim_cocuk: { name: 'Cocuk Giyim', rate: 17.0 },
      ayakkabi: { name: 'Ayakkabi', rate: 17.0 },
      canta_aksesuar: { name: 'Canta & Aksesuar', rate: 17.0 },
      kozmetik: { name: 'Kozmetik & Kisisel Bakim', rate: 13.5 },
      ev_yasam: { name: 'Ev & Yasam', rate: 14.5 },
      ev_mobilya: { name: 'Mobilya', rate: 11.5 },
      ev_dekorasyon: { name: 'Dekorasyon', rate: 14.5 },
      mutfak: { name: 'Mutfak Gerecleri', rate: 14.5 },
      spor_outdoor: { name: 'Spor & Outdoor', rate: 13.5 },
      oyuncak: { name: 'Oyuncak', rate: 14.5 },
      kitap: { name: 'Kitap & Kirtasiye', rate: 15.5 },
      anne_bebek: { name: 'Anne & Bebek', rate: 13.5 },
      pet_urunleri: { name: 'Pet Urunleri', rate: 13.5 },
      oto_aksesuar: { name: 'Oto Aksesuar', rate: 13.5 },
      gida: { name: 'Gida & Icecek', rate: 11.5 },
      saat_takı: { name: 'Saat & Taki', rate: 15.5 },
      supermarket: { name: 'Supermarket', rate: 9.5 }
    }
  });

  /** Plan yapilandirmalari */
  const PLAN_CONFIG = Object.freeze({
    free: { label: 'Ucretsiz', icon: '\uD83C\uDD93', cssClass: 'pr-plan-free' },
    pro: { label: 'PRO', icon: '\uD83D\uDE80', cssClass: 'pr-plan-pro' },
    business: { label: 'Business', icon: '\uD83C\uDFC6', cssClass: 'pr-plan-business' }
  });

  /** Kategori ikonlari */
  const CATEGORY_ICONS = Object.freeze({
    elektronik: '\uD83D\uDCF1',
    giyim: '\uD83D\uDC55',
    ayakkabi: '\uD83D\uDC5F',
    canta: '\uD83D\uDC5C',
    kozmetik: '\uD83D\uDC84',
    ev: '\uD83C\uDFE0',
    mobilya: '\uD83D\uDECB\uFE0F',
    spor: '\u26BD',
    oyuncak: '\uD83E\uDDF8',
    kitap: '\uD83D\uDCDA',
    anne: '\uD83D\uDC76',
    pet: '\uD83D\uDC3E',
    oto: '\uD83D\uDE97',
    gida: '\uD83C\uDF4E',
    saat: '\u231A',
    supermarket: '\uD83D\uDED2',
    mutfak: '\uD83C\uDF73',
    dekorasyon: '\uD83C\uDFA8',
    default: '\uD83D\uDCE6'
  });

  /** Skor renkleri */
  const SCORE_COLORS = Object.freeze({
    excellent: '#16a34a',
    good: '#22c55e',
    moderate: '#f97316',
    risky: '#ef4444',
    avoid: '#991b1b'
  });

  // ─── State ──────────────────────────────────────────────────

  /** Aktif platform sekmesi */
  let activePlatform = 'trendyol';

  /** Mevcut urun verisi */
  let currentProductData = null;

  /** Kullanici ayarlari */
  let userSettings = {
    searchOverlay: true,
    productPanel: true,
    notifications: true,
    defaultCost: null,
    plan: 'free',
    licenseKey: ''
  };

  // ─── Yardimci Fonksiyonlar ──────────────────────────────────

  /**
   * Sayiyi 2 ondalik basamaga yuvarlar.
   * @param {number} value
   * @returns {number}
   */
  function round2(value) {
    return Math.round((value + Number.EPSILON) * 100) / 100;
  }

  /**
   * Sayiyi Turkce TL formatinda gosterir.
   * @param {number} value
   * @param {boolean} [showCurrency]
   * @returns {string}
   */
  function formatTL(value, showCurrency) {
    if (value == null || Number.isNaN(value)) {
      return showCurrency !== false ? '-- TL' : '--';
    }

    const formatted = new Intl.NumberFormat('tr-TR', {
      minimumFractionDigits: 2,
      maximumFractionDigits: 2
    }).format(value);

    return showCurrency !== false ? formatted + ' TL' : formatted;
  }

  /**
   * Yuzde degerini formatlar.
   * @param {number} value
   * @returns {string}
   */
  function formatPercent(value) {
    if (value == null || Number.isNaN(value)) return '--%';
    return '%' + new Intl.NumberFormat('tr-TR', {
      minimumFractionDigits: 1,
      maximumFractionDigits: 1
    }).format(value);
  }

  /**
   * Buyuk sayilari Turkce kisa formatta gosterir (134K, 2.3M).
   * @param {number} value
   * @returns {string}
   */
  function formatShortNumber(value) {
    if (value == null || Number.isNaN(value)) return '0';
    const abs = Math.abs(value);
    if (abs >= 1000000) {
      const m = value / 1000000;
      return (m % 1 === 0 ? String(m) : m.toFixed(1)) + 'M';
    }
    if (abs >= 1000) {
      const k = value / 1000;
      return (k % 1 === 0 ? String(k) : k.toFixed(1)) + 'K';
    }
    return String(Math.round(value));
  }

  /**
   * Guvenli DOM element olusturucu.
   * @param {string} tag
   * @param {Object} [attrs]
   * @param {Array<Node|string>} [children]
   * @returns {HTMLElement}
   */
  function el(tag, attrs, children) {
    const element = document.createElement(tag);

    if (attrs) {
      for (const [key, value] of Object.entries(attrs)) {
        if (key === 'className') {
          element.className = value;
        } else if (key === 'textContent') {
          element.textContent = value;
        } else if (key === 'dataset') {
          for (const [dk, dv] of Object.entries(value)) {
            element.dataset[dk] = dv;
          }
        } else if (key.startsWith('on') && typeof value === 'function') {
          element.addEventListener(key.slice(2).toLowerCase(), value);
        } else {
          element.setAttribute(key, value);
        }
      }
    }

    if (children) {
      for (const child of children) {
        if (typeof child === 'string') {
          element.appendChild(document.createTextNode(child));
        } else if (child instanceof Node) {
          element.appendChild(child);
        }
      }
    }

    return element;
  }

  /**
   * Container icindeki tum child node'lari temizler.
   * @param {HTMLElement} container
   */
  function clearChildren(container) {
    while (container.firstChild) {
      container.removeChild(container.firstChild);
    }
  }

  /**
   * Kategori anahtarindan ikon secer.
   * @param {string} categoryKey
   * @returns {string}
   */
  function getCategoryIcon(categoryKey) {
    if (!categoryKey) return CATEGORY_ICONS.default;
    const key = categoryKey.toLowerCase();
    for (const [pattern, icon] of Object.entries(CATEGORY_ICONS)) {
      if (key.includes(pattern)) return icon;
    }
    return CATEGORY_ICONS.default;
  }

  /**
   * background.js ile mesajlasma.
   * @param {Object} message
   * @returns {Promise<Object>}
   */
  function sendMessage(message) {
    return new Promise(function (resolve) {
      try {
        chrome.runtime.sendMessage(message, function (response) {
          if (chrome.runtime.lastError) {
            resolve({ success: false, error: chrome.runtime.lastError.message });
            return;
          }
          resolve(response || { success: false, error: 'Yanit alinamadi' });
        });
      } catch (err) {
        resolve({ success: false, error: err.message });
      }
    });
  }

  /**
   * Skor degerinden renk belirler.
   * @param {number} score - 0-10 arasi
   * @returns {string} CSS renk kodu
   */
  function getScoreColor(score) {
    if (score >= 8) return SCORE_COLORS.excellent;
    if (score >= 6.5) return SCORE_COLORS.good;
    if (score >= 4.5) return SCORE_COLORS.moderate;
    if (score >= 3) return SCORE_COLORS.risky;
    return SCORE_COLORS.avoid;
  }

  // ─── Yerel Kar Hesaplama ────────────────────────────────────

  /**
   * Tek platform icin kar hesaplama (panel icin yerel kopya).
   * @param {string} platform
   * @param {string} categoryKey
   * @param {number} salePrice
   * @param {number|null} costPrice
   * @param {number} [cargoWeight]
   * @returns {Object}
   */
  function calculateProfit(platform, categoryKey, salePrice, costPrice, cargoWeight) {
    const config = PLATFORM_CONFIG[platform];
    if (!config) return null;

    const platformRates = COMMISSION_RATES[platform];
    const categoryData = platformRates ? platformRates[categoryKey] : null;
    const commissionRate = categoryData ? categoryData.rate : 0;

    const kdvMultiplier = 1 + (DEFAULT_KDV_RATE / 100);
    const salePriceExKDV = round2(salePrice / kdvMultiplier);

    let commissionAmount;
    if (config.commissionBase === 'kdv_dahil') {
      commissionAmount = round2(salePrice * commissionRate / 100);
    } else {
      commissionAmount = round2(salePriceExKDV * commissionRate / 100);
    }

    const commissionKDV = round2(commissionAmount * SERVICE_KDV_RATE);
    const platformFee = config.platformFee;
    const platformFeeKDV = round2(platformFee * SERVICE_KDV_RATE);

    const weight = Math.max(cargoWeight || 1, 1);
    const estimatedCargo = round2(Math.max(weight * config.defaultCargoPerDesi, config.minCargo));

    const totalCosts = round2(
      commissionAmount + commissionKDV + platformFee + platformFeeKDV + estimatedCargo
    );

    const netRevenue = round2(salePrice - totalCosts);
    const netProfit = (costPrice != null) ? round2(netRevenue - costPrice) : null;

    let netMarginPercent = null;
    if (netProfit != null && salePrice > 0) {
      netMarginPercent = round2((netProfit / salePrice) * 100);
    }

    return {
      platform: platform,
      platformName: config.name,
      salePrice: round2(salePrice),
      commissionRate: commissionRate,
      commissionAmount: commissionAmount,
      commissionKDV: commissionKDV,
      platformFee: round2(platformFee),
      platformFeeKDV: platformFeeKDV,
      estimatedCargo: estimatedCargo,
      totalCosts: totalCosts,
      netRevenue: netRevenue,
      netProfit: netProfit,
      netMarginPercent: netMarginPercent
    };
  }

  /**
   * 3 platform icin karsilastirmali hesaplama.
   * @param {string} categoryKey
   * @param {number} salePrice
   * @param {number|null} costPrice
   * @param {number} [cargoWeight]
   * @returns {Object}
   */
  function compareAllPlatforms(categoryKey, salePrice, costPrice, cargoWeight) {
    const results = {};
    let bestPlatform = '';
    let bestNetRevenue = -Infinity;

    for (const platform of VALID_PLATFORMS) {
      const result = calculateProfit(platform, categoryKey, salePrice, costPrice, cargoWeight);
      results[platform] = result;
      if (result && result.netRevenue > bestNetRevenue) {
        bestNetRevenue = result.netRevenue;
        bestPlatform = platform;
      }
    }

    return {
      trendyol: results.trendyol || null,
      hepsiburada: results.hepsiburada || null,
      n11: results.n11 || null,
      best: bestPlatform
    };
  }

  // ─── DOM Referanslari ───────────────────────────────────────

  /** @type {HTMLElement} */
  let productSection;
  /** @type {HTMLElement} */
  let productContent;
  /** @type {HTMLElement} */
  let recentContent;
  /** @type {HTMLElement} */
  let alertsContent;
  /** @type {HTMLElement} */
  let settingsSection;
  /** @type {HTMLElement} */
  let settingsContent;
  /** @type {HTMLElement} */
  let licenseSection;
  /** @type {HTMLElement} */
  let licenseContent;
  /** @type {HTMLElement} */
  let planContent;

  // ─── Urun Analizi Render ────────────────────────────────────

  /**
   * Tam urun analiz gorunumunu render eder.
   * Content script'ten gelen verileri isler.
   * @param {Object} data - Urun verisi
   */
  function renderProductAnalysis(data) {
    if (!data) {
      productSection.classList.add('pr-hidden');
      return;
    }

    currentProductData = data;
    productSection.classList.remove('pr-hidden');
    clearChildren(productContent);

    const card = el('div', { className: 'pr-product-card pr-fade-in' });

    // Urun basligi
    const header = el('div', { className: 'pr-product-header' });
    header.appendChild(el('div', {
      className: 'pr-product-name',
      textContent: data.productName || data.name || 'Urun'
    }));

    const metaParts = [];
    if (data.brand) metaParts.push(data.brand);
    if (data.categoryName || data.category) metaParts.push(data.categoryName || data.category);
    if (metaParts.length > 0) {
      header.appendChild(el('div', {
        className: 'pr-product-meta',
        textContent: metaParts.join(' \u00B7 ')
      }));
    }
    card.appendChild(header);

    // Urun Skoru (varsa)
    if (data.score != null || data.nicheScore != null) {
      const scoreValue = data.score || (data.nicheScore ? data.nicheScore.overall : null);
      if (scoreValue != null) {
        card.appendChild(renderScoreBlock(scoreValue));
      }
    }

    // Satis tahmin istatistikleri
    if (data.salesEstimate || data.estimatedMonthlySales != null) {
      card.appendChild(renderSalesStats(data));
    }

    // Kar hesaplama doekuemue
    const salePrice = data.price || data.salePrice || 0;
    const platform = data.platform || activePlatform;
    const categoryKey = data.categoryKey || '';

    if (salePrice > 0) {
      card.appendChild(renderProfitBreakdown(platform, categoryKey, salePrice, data.costPrice || null));
    }

    // Maliyet girisi
    card.appendChild(renderCostInput(platform, categoryKey, salePrice));

    // Platform karsilastirma
    if (salePrice > 0 && categoryKey) {
      card.appendChild(renderPlatformComparison(categoryKey, salePrice, data.costPrice || null));
    }

    // Fiyat gecmisi
    card.appendChild(renderPriceHistory(data.priceHistory || null, salePrice));

    productContent.appendChild(card);
  }

  /**
   * Urun skoru blogu olusturur.
   * @param {number} score - 0-10 arasi
   * @returns {HTMLElement}
   */
  function renderScoreBlock(score) {
    const color = getScoreColor(score);
    const percentage = (score / 10) * 100;

    const block = el('div', { className: 'pr-score-block' });

    const valueContainer = el('div');
    const valueEl = el('span', { className: 'pr-score-value' });
    valueEl.textContent = score.toFixed(1);
    valueEl.style.color = color;
    valueContainer.appendChild(valueEl);
    valueContainer.appendChild(el('span', { className: 'pr-score-max', textContent: '/10' }));
    block.appendChild(valueContainer);

    const barContainer = el('div', { className: 'pr-score-bar-container' });
    barContainer.appendChild(el('div', { className: 'pr-score-label', textContent: 'Urun Skoru' }));

    const bar = el('div', { className: 'pr-score-bar' });
    const fill = el('div', { className: 'pr-score-bar-fill' });
    fill.style.width = percentage + '%';
    fill.style.background = color;
    bar.appendChild(fill);
    barContainer.appendChild(bar);

    block.appendChild(barContainer);
    return block;
  }

  /**
   * Satis tahmini istatistiklerini render eder.
   * @param {Object} data
   * @returns {HTMLElement}
   */
  function renderSalesStats(data) {
    const estimate = data.salesEstimate || {};
    const grid = el('div', { className: 'pr-stat-grid' });

    const monthlySales = estimate.estimatedMonthlySales || data.estimatedMonthlySales || 0;
    const monthlyRevenue = estimate.estimatedMonthlyRevenue || data.estimatedMonthlyRevenue || 0;
    const reviewVelocity = data.reviewVelocity || estimate.reviewVelocity || 0;
    const favoriteCount = data.favoriteCount || 0;

    // Tahmini satis
    const salesCard = el('div', { className: 'pr-stat-card' });
    salesCard.appendChild(el('div', { className: 'pr-stat-card-label', textContent: 'Tahmini Satis' }));
    salesCard.appendChild(el('div', {
      className: 'pr-stat-card-value',
      textContent: estimate.salesText || ('~' + monthlySales + '/ay')
    }));
    grid.appendChild(salesCard);

    // Tahmini ciro
    const revenueCard = el('div', { className: 'pr-stat-card' });
    revenueCard.appendChild(el('div', { className: 'pr-stat-card-label', textContent: 'Tahmini Ciro' }));
    revenueCard.appendChild(el('div', {
      className: 'pr-stat-card-value',
      textContent: estimate.revenueText || ('~' + formatShortNumber(monthlyRevenue) + ' TL/ay')
    }));
    grid.appendChild(revenueCard);

    // Yorum hizi
    const reviewCard = el('div', { className: 'pr-stat-card' });
    reviewCard.appendChild(el('div', { className: 'pr-stat-card-label', textContent: 'Yorum Hizi' }));
    reviewCard.appendChild(el('div', {
      className: 'pr-stat-card-value',
      textContent: reviewVelocity + '/ay'
    }));
    grid.appendChild(reviewCard);

    // Favori
    const favCard = el('div', { className: 'pr-stat-card' });
    favCard.appendChild(el('div', { className: 'pr-stat-card-label', textContent: 'Favori' }));
    favCard.appendChild(el('div', {
      className: 'pr-stat-card-value',
      textContent: formatShortNumber(favoriteCount)
    }));
    grid.appendChild(favCard);

    return grid;
  }

  /**
   * Kar hesaplama detay doekuemue olusturur.
   * @param {string} platform
   * @param {string} categoryKey
   * @param {number} salePrice
   * @param {number|null} costPrice
   * @returns {HTMLElement}
   */
  function renderProfitBreakdown(platform, categoryKey, salePrice, costPrice) {
    const result = calculateProfit(platform, categoryKey, salePrice, costPrice);
    if (!result) return el('div');

    const container = el('div', { className: 'pr-breakdown' });
    container.appendChild(el('div', { className: 'pr-breakdown-title', textContent: 'Kar Hesaplama' }));

    // Satis fiyati
    const saleRow = el('div', { className: 'pr-breakdown-row' });
    saleRow.appendChild(el('span', { className: 'pr-breakdown-label', textContent: 'Satis' }));
    saleRow.appendChild(el('span', { className: 'pr-breakdown-value', textContent: formatTL(salePrice) }));
    container.appendChild(saleRow);

    // Komisyon
    const commRow = el('div', { className: 'pr-breakdown-row' });
    commRow.appendChild(el('span', {
      className: 'pr-breakdown-label',
      textContent: 'Komisyon (%' + result.commissionRate + ')'
    }));
    commRow.appendChild(el('span', {
      className: 'pr-breakdown-value pr-negative',
      textContent: '-' + formatTL(result.commissionAmount)
    }));
    container.appendChild(commRow);

    // Komisyon KDV
    const commKdvRow = el('div', { className: 'pr-breakdown-row' });
    commKdvRow.appendChild(el('span', { className: 'pr-breakdown-label', textContent: 'Komisyon KDV' }));
    commKdvRow.appendChild(el('span', {
      className: 'pr-breakdown-value pr-negative',
      textContent: '-' + formatTL(result.commissionKDV)
    }));
    container.appendChild(commKdvRow);

    // Platform hizmet bedeli
    const platRow = el('div', { className: 'pr-breakdown-row' });
    platRow.appendChild(el('span', { className: 'pr-breakdown-label', textContent: 'Platform Hizmeti' }));
    platRow.appendChild(el('span', {
      className: 'pr-breakdown-value pr-negative',
      textContent: '-' + formatTL(result.platformFee + result.platformFeeKDV)
    }));
    container.appendChild(platRow);

    // Kargo
    const cargoRow = el('div', { className: 'pr-breakdown-row' });
    cargoRow.appendChild(el('span', { className: 'pr-breakdown-label', textContent: 'Kargo' }));
    cargoRow.appendChild(el('span', {
      className: 'pr-breakdown-value pr-negative',
      textContent: '-' + formatTL(result.estimatedCargo)
    }));
    container.appendChild(cargoRow);

    // Ayirici
    container.appendChild(el('div', { className: 'pr-breakdown-separator' }));

    // Net gelir
    const netRow = el('div', { className: 'pr-breakdown-row' });
    netRow.appendChild(el('span', { className: 'pr-breakdown-label', textContent: 'Net Gelir' }));
    netRow.appendChild(el('span', {
      className: 'pr-breakdown-value',
      textContent: formatTL(result.netRevenue)
    }));
    container.appendChild(netRow);

    return container;
  }

  /**
   * Maliyet girisi ve net kar sonucunu olusturur.
   * @param {string} platform
   * @param {string} categoryKey
   * @param {number} salePrice
   * @returns {HTMLElement}
   */
  function renderCostInput(platform, categoryKey, salePrice) {
    const container = el('div');

    // Maliyet girdisi
    const inputRow = el('div', { className: 'pr-inline-input-row' });
    inputRow.appendChild(el('span', { className: 'pr-inline-label', textContent: 'Maliyetiniz:' }));

    const inputWrap = el('div', { className: 'pr-inline-input-wrap' });
    const input = el('input', {
      type: 'number',
      className: 'pr-inline-input',
      id: 'prPanelCostInput',
      placeholder: '0,00',
      min: '0',
      step: '0.01'
    });

    // Varsayilan maliyet varsa doldur
    if (currentProductData && currentProductData.costPrice != null) {
      input.value = currentProductData.costPrice;
    } else if (userSettings.defaultCost != null && userSettings.defaultCost > 0) {
      input.value = userSettings.defaultCost;
    }

    inputWrap.appendChild(input);
    inputWrap.appendChild(el('span', { className: 'pr-inline-suffix', textContent: 'TL' }));
    inputRow.appendChild(inputWrap);
    container.appendChild(inputRow);

    // Sonuc alani
    const profitResultEl = el('div', { className: 'pr-profit-result', id: 'prPanelProfitResult' });
    container.appendChild(profitResultEl);

    // Input change handler
    input.addEventListener('input', function () {
      const costPrice = parseFloat(input.value);
      updateProfitResult(platform, categoryKey, salePrice, costPrice);
    });

    // Ilk hesaplama
    const initialCost = parseFloat(input.value);
    if (initialCost > 0) {
      // Render sonrasi hesapla
      setTimeout(function () {
        updateProfitResult(platform, categoryKey, salePrice, initialCost);
      }, 0);
    }

    return container;
  }

  /**
   * Net kar sonucunu gunceller.
   * @param {string} platform
   * @param {string} categoryKey
   * @param {number} salePrice
   * @param {number} costPrice
   */
  function updateProfitResult(platform, categoryKey, salePrice, costPrice) {
    const resultEl = document.getElementById('prPanelProfitResult');
    if (!resultEl) return;

    clearChildren(resultEl);

    if (!costPrice || costPrice <= 0 || !salePrice || salePrice <= 0) return;

    const result = calculateProfit(platform, categoryKey, salePrice, costPrice);
    if (!result || result.netProfit == null) return;

    resultEl.appendChild(el('span', { className: 'pr-profit-label', textContent: 'Net Kar:' }));

    const valueClass = result.netProfit >= 0 ? 'pr-positive' : 'pr-negative';
    resultEl.appendChild(el('span', {
      className: 'pr-profit-value ' + valueClass,
      textContent: formatTL(result.netProfit)
    }));

    if (result.netMarginPercent != null) {
      const badgeClass = result.netMarginPercent >= 0 ? 'pr-positive' : 'pr-negative';
      const statusIcon = result.netMarginPercent >= 15 ? ' \u2705' : result.netMarginPercent >= 0 ? ' \u2705' : ' \u274C';
      resultEl.appendChild(el('span', {
        className: 'pr-profit-badge ' + badgeClass,
        textContent: formatPercent(result.netMarginPercent) + statusIcon
      }));
    }

    // Platform karsilastirmayi da guncelle
    const compEl = document.getElementById('prPanelPlatformComparison');
    if (compEl && categoryKey) {
      clearChildren(compEl);
      const comp = renderPlatformComparisonRows(categoryKey, salePrice, costPrice);
      if (comp) compEl.appendChild(comp);
    }
  }

  /**
   * Platform karsilastirma bolumunu olusturur.
   * @param {string} categoryKey
   * @param {number} salePrice
   * @param {number|null} costPrice
   * @returns {HTMLElement}
   */
  function renderPlatformComparison(categoryKey, salePrice, costPrice) {
    const container = el('div', { className: 'pr-breakdown' });
    container.appendChild(el('div', { className: 'pr-breakdown-title', textContent: 'Platform Karsilastirma' }));

    const rowsContainer = el('div', { id: 'prPanelPlatformComparison' });
    const rows = renderPlatformComparisonRows(categoryKey, salePrice, costPrice);
    if (rows) rowsContainer.appendChild(rows);
    container.appendChild(rowsContainer);

    return container;
  }

  /**
   * Karsilastirma satirlarini olusturur.
   * @param {string} categoryKey
   * @param {number} salePrice
   * @param {number|null} costPrice
   * @returns {HTMLElement|null}
   */
  function renderPlatformComparisonRows(categoryKey, salePrice, costPrice) {
    const comparison = compareAllPlatforms(categoryKey, salePrice, costPrice);
    const wrapper = el('div');

    for (const platform of VALID_PLATFORMS) {
      const result = comparison[platform];
      if (!result) continue;

      const isBest = (platform === comparison.best);
      const rowClass = 'pr-comparison-row' + (isBest ? ' pr-best' : '');
      const row = el('div', { className: rowClass });

      row.appendChild(el('span', {
        className: 'pr-comparison-platform',
        textContent: result.platformName
      }));

      const displayValue = result.netProfit != null ? result.netProfit : result.netRevenue;
      const valueClass = displayValue >= 0 ? 'pr-text-positive' : 'pr-text-negative';
      row.appendChild(el('span', {
        className: 'pr-comparison-profit ' + valueClass,
        textContent: formatTL(displayValue)
      }));

      if (result.netMarginPercent != null) {
        row.appendChild(el('span', {
          className: 'pr-comparison-margin',
          textContent: formatPercent(result.netMarginPercent)
        }));
      }

      if (isBest) {
        row.appendChild(el('span', { className: 'pr-comparison-best-icon', textContent: '\u2605' }));
      }

      wrapper.appendChild(row);
    }

    return wrapper;
  }

  /**
   * Fiyat gecmisi bolumunu olusturur.
   * @param {Array|null} priceHistory
   * @param {number} currentPrice
   * @returns {HTMLElement}
   */
  function renderPriceHistory(priceHistory, currentPrice) {
    const container = el('div', { className: 'pr-breakdown' });
    container.appendChild(el('div', { className: 'pr-breakdown-title', textContent: 'Fiyat Gecmisi' }));

    const chartContainer = el('div', { className: 'pr-price-history-chart' });

    if (!priceHistory || priceHistory.length === 0) {
      chartContainer.appendChild(el('span', {
        className: 'pr-price-history-placeholder',
        textContent: 'Fiyat gecmisi verisi bekleniyor...'
      }));
      container.appendChild(chartContainer);

      // Mevcut fiyat bilgisi (sadece current price varsa)
      if (currentPrice > 0) {
        const statsGrid = el('div', { className: 'pr-price-stats' });
        statsGrid.appendChild(createPriceStat('Simdi', formatTL(currentPrice)));
        container.appendChild(statsGrid);
      }

      return container;
    }

    // Basit SVG cizgi grafigi olustur
    const prices = priceHistory.map(function (item) {
      return typeof item === 'number' ? item : (item.price || 0);
    });

    const minPrice = Math.min.apply(null, prices);
    const maxPrice = Math.max.apply(null, prices);
    const avgPrice = round2(prices.reduce(function (s, p) { return s + p; }, 0) / prices.length);

    // SVG olustur
    chartContainer.appendChild(createPriceChart(prices, minPrice, maxPrice));
    container.appendChild(chartContainer);

    // Istatistik gridi
    const statsGrid = el('div', { className: 'pr-price-stats' });
    statsGrid.appendChild(createPriceStat('Min', formatTL(minPrice)));
    statsGrid.appendChild(createPriceStat('Ort', formatTL(avgPrice)));
    statsGrid.appendChild(createPriceStat('Max', formatTL(maxPrice)));
    statsGrid.appendChild(createPriceStat('Simdi', formatTL(currentPrice)));
    container.appendChild(statsGrid);

    return container;
  }

  /**
   * Fiyat istatistigi elementi olusturur.
   * @param {string} label
   * @param {string} value
   * @returns {HTMLElement}
   */
  function createPriceStat(label, value) {
    return el('div', { className: 'pr-price-stat' }, [
      el('span', { className: 'pr-price-stat-label', textContent: label + ':' }),
      el('span', { className: 'pr-price-stat-value', textContent: value })
    ]);
  }

  /**
   * Basit SVG cizgi grafigi olusturur.
   * @param {Array<number>} prices
   * @param {number} minPrice
   * @param {number} maxPrice
   * @returns {SVGElement}
   */
  function createPriceChart(prices, minPrice, maxPrice) {
    const svgNS = 'http://www.w3.org/2000/svg';
    const svg = document.createElementNS(svgNS, 'svg');
    svg.setAttribute('viewBox', '0 0 300 70');
    svg.setAttribute('preserveAspectRatio', 'none');

    const range = maxPrice - minPrice || 1;
    const padding = 5;
    const chartWidth = 300 - padding * 2;
    const chartHeight = 70 - padding * 2;

    // Cizgi path olustur
    const points = prices.map(function (price, index) {
      const x = padding + (index / (prices.length - 1 || 1)) * chartWidth;
      const y = padding + chartHeight - ((price - minPrice) / range) * chartHeight;
      return x + ',' + y;
    });

    // Gradient alan
    const areaPath = document.createElementNS(svgNS, 'path');
    const areaD = 'M' + points[0] + ' L' + points.join(' L') +
      ' L' + (padding + chartWidth) + ',' + (padding + chartHeight) +
      ' L' + padding + ',' + (padding + chartHeight) + ' Z';
    areaPath.setAttribute('d', areaD);
    areaPath.setAttribute('fill', 'rgba(79, 70, 229, 0.08)');
    svg.appendChild(areaPath);

    // Cizgi
    const linePath = document.createElementNS(svgNS, 'polyline');
    linePath.setAttribute('points', points.join(' '));
    linePath.setAttribute('fill', 'none');
    linePath.setAttribute('stroke', '#4f46e5');
    linePath.setAttribute('stroke-width', '2');
    linePath.setAttribute('stroke-linecap', 'round');
    linePath.setAttribute('stroke-linejoin', 'round');
    svg.appendChild(linePath);

    // Son nokta
    const lastPoint = points[points.length - 1].split(',');
    const circle = document.createElementNS(svgNS, 'circle');
    circle.setAttribute('cx', lastPoint[0]);
    circle.setAttribute('cy', lastPoint[1]);
    circle.setAttribute('r', '3');
    circle.setAttribute('fill', '#4f46e5');
    svg.appendChild(circle);

    return svg;
  }

  // ─── Son Analizler ──────────────────────────────────────────

  /**
   * Son analizleri yuklayip render eder.
   */
  async function renderRecentAnalyses() {
    const response = await sendMessage({ action: 'get_recent_analyses' });

    clearChildren(recentContent);

    if (!response || !response.success || !response.data || response.data.length === 0) {
      renderEmptyState(recentContent, '\uD83D\uDD0D', 'Henuz analiz yapilmadi. Bir urun sayfasini ziyaret edin.');
      return;
    }

    const list = el('div', { className: 'pr-analysis-list pr-fade-in' });

    for (const item of response.data.slice(0, 10)) {
      const icon = getCategoryIcon(item.categoryKey || item.category || '');
      const marginValue = item.netMarginPercent || item.margin || 0;
      const platformName = item.platformName || (PLATFORM_CONFIG[item.platform] ? PLATFORM_CONFIG[item.platform].name : '') || item.platform || '';

      const row = el('div', { className: 'pr-analysis-item' });

      row.appendChild(el('span', { className: 'pr-analysis-icon', textContent: icon }));

      const info = el('div', { className: 'pr-analysis-info' });
      info.appendChild(el('div', {
        className: 'pr-analysis-name',
        textContent: item.productName || item.name || 'Urun'
      }));
      info.appendChild(el('div', {
        className: 'pr-analysis-platform',
        textContent: platformName
      }));
      row.appendChild(info);

      const marginClass = marginValue >= 0 ? 'pr-text-positive' : 'pr-text-negative';
      row.appendChild(el('span', {
        className: 'pr-analysis-margin ' + marginClass,
        textContent: formatPercent(marginValue)
      }));

      list.appendChild(row);
    }

    recentContent.appendChild(list);
  }

  // ─── Fiyat Alarmlari ───────────────────────────────────────

  /**
   * Fiyat alarmlarini yuklayip render eder.
   */
  async function renderPriceAlerts() {
    const response = await sendMessage({ action: 'get_price_alerts' });

    clearChildren(alertsContent);

    if (!response || !response.success || !response.data || response.data.length === 0) {
      renderEmptyState(alertsContent, '\uD83D\uDD14', 'Henuz fiyat alarmi yok.');
      return;
    }

    const container = el('div', { className: 'pr-fade-in' });

    for (const alert of response.data.slice(0, 5)) {
      const item = el('div', { className: 'pr-alert-item' });

      const changeType = alert.changePercent >= 0 ? 'up' : 'down';
      const changeIcon = changeType === 'up' ? '\u2B06\uFE0F' : '\u2B07\uFE0F';

      item.appendChild(el('span', { className: 'pr-alert-icon', textContent: changeIcon }));

      const info = el('div', { className: 'pr-alert-info' });
      info.appendChild(el('div', {
        className: 'pr-alert-name',
        textContent: alert.productName || 'Urun'
      }));
      info.appendChild(el('div', {
        className: 'pr-alert-detail',
        textContent: formatTL(alert.oldPrice) + ' \u2192 ' + formatTL(alert.newPrice)
      }));
      item.appendChild(info);

      const changeClass = 'pr-alert-change pr-' + changeType;
      const changeText = (alert.changePercent >= 0 ? '+' : '') + formatPercent(alert.changePercent);
      item.appendChild(el('span', { className: changeClass, textContent: changeText }));

      container.appendChild(item);
    }

    alertsContent.appendChild(container);
  }

  // ─── Ayarlar ────────────────────────────────────────────────

  /**
   * Ayarlar panelini render eder.
   */
  function renderSettings() {
    clearChildren(settingsContent);

    const container = el('div', { className: 'pr-fade-in' });

    // Arama overlay
    container.appendChild(createSettingToggle(
      'Arama Overlay',
      'searchOverlay',
      userSettings.searchOverlay
    ));

    // Urun paneli
    container.appendChild(createSettingToggle(
      'Urun Paneli',
      'productPanel',
      userSettings.productPanel
    ));

    // Bildirimler
    container.appendChild(createSettingToggle(
      'Bildirimler',
      'notifications',
      userSettings.notifications
    ));

    // Varsayilan maliyet
    const costRow = el('div', { className: 'pr-settings-input-row' });
    costRow.appendChild(el('span', {
      className: 'pr-settings-label',
      textContent: 'Varsayilan Maliyet:'
    }));

    const costWrap = el('div', { className: 'pr-inline-input-wrap' });
    const costInput = el('input', {
      type: 'number',
      className: 'pr-settings-input',
      id: 'prDefaultCostInput',
      placeholder: '0',
      min: '0',
      step: '0.01',
      value: userSettings.defaultCost || ''
    });
    costInput.addEventListener('change', function () {
      const value = parseFloat(costInput.value);
      handleSettingsChange('defaultCost', value > 0 ? value : null);
    });
    costWrap.appendChild(costInput);
    costWrap.appendChild(el('span', { className: 'pr-inline-suffix', textContent: 'TL' }));
    costRow.appendChild(costWrap);
    container.appendChild(costRow);

    settingsContent.appendChild(container);
  }

  /**
   * Toggle ayar satiri olusturur.
   * @param {string} label
   * @param {string} settingKey
   * @param {boolean} currentValue
   * @returns {HTMLElement}
   */
  function createSettingToggle(label, settingKey, currentValue) {
    const row = el('div', { className: 'pr-settings-row' });
    row.appendChild(el('span', { className: 'pr-settings-label', textContent: label }));

    const toggle = el('label', { className: 'pr-toggle' });
    const checkbox = el('input', { type: 'checkbox' });
    if (currentValue) checkbox.checked = true;
    checkbox.addEventListener('change', function () {
      handleSettingsChange(settingKey, checkbox.checked);
    });
    toggle.appendChild(checkbox);
    toggle.appendChild(el('span', { className: 'pr-toggle-slider' }));
    row.appendChild(toggle);

    return row;
  }

  /**
   * Ayar degisikligini background'a gonderir.
   * @param {string} key
   * @param {*} value
   */
  async function handleSettingsChange(key, value) {
    userSettings = Object.assign({}, userSettings, { [key]: value });

    await sendMessage({
      action: 'update_settings',
      data: { key: key, value: value }
    });
  }

  // ─── Lisans ─────────────────────────────────────────────────

  /**
   * Lisans bolumunu render eder.
   */
  function renderLicenseSection() {
    clearChildren(licenseContent);

    const container = el('div', { className: 'pr-fade-in' });

    // Mevcut plan
    const planConfig = PLAN_CONFIG[userSettings.plan] || PLAN_CONFIG.free;
    container.appendChild(el('div', { textContent: 'Plan: ' + planConfig.icon + ' ' + planConfig.label }, []));

    // Lisans girdisi
    const licenseRow = el('div', { className: 'pr-license-row' });

    const licenseInput = el('input', {
      type: 'text',
      className: 'pr-license-input',
      id: 'prLicenseInput',
      placeholder: 'XXXX-XXXX-XXXX-XXXX',
      maxlength: '19',
      value: userSettings.licenseKey || ''
    });
    licenseRow.appendChild(licenseInput);

    const activateBtn = el('button', {
      className: 'pr-activate-btn',
      id: 'prActivateBtn',
      textContent: 'Aktive Et'
    });
    activateBtn.addEventListener('click', function () {
      handleLicenseActivation(licenseInput.value);
    });
    licenseRow.appendChild(activateBtn);

    container.appendChild(licenseRow);

    // Durum mesaji alani
    container.appendChild(el('div', {
      className: 'pr-license-status pr-hidden',
      id: 'prLicenseStatus'
    }));

    licenseContent.appendChild(container);
  }

  /**
   * Lisans anahtarini dogrulayip aktive eder.
   * @param {string} licenseKey
   */
  async function handleLicenseActivation(licenseKey) {
    const statusEl = document.getElementById('prLicenseStatus');
    const activateBtn = document.getElementById('prActivateBtn');

    if (!licenseKey || licenseKey.trim().length < 10) {
      if (statusEl) {
        statusEl.textContent = 'Gecerli bir lisans anahtari girin.';
        statusEl.className = 'pr-license-status pr-error';
      }
      return;
    }

    // Butonu devre disi birak
    if (activateBtn) {
      activateBtn.disabled = true;
      activateBtn.textContent = 'Kontrol ediliyor...';
    }

    const response = await sendMessage({
      action: 'activate_license',
      data: { licenseKey: licenseKey.trim() }
    });

    if (activateBtn) {
      activateBtn.disabled = false;
      activateBtn.textContent = 'Aktive Et';
    }

    if (statusEl) {
      if (response && response.success) {
        statusEl.textContent = 'Lisans basariyla aktive edildi! Plan: ' + (response.data?.plan || 'PRO');
        statusEl.className = 'pr-license-status pr-success';

        // Ayarlari guncelle
        userSettings = Object.assign({}, userSettings, {
          plan: response.data?.plan || 'pro',
          licenseKey: licenseKey.trim()
        });

        // Plan bolumunu yeniden render et
        loadPlanInfo();
      } else {
        statusEl.textContent = response?.error || 'Lisans dogrulanamadi. Lutfen kontrol edin.';
        statusEl.className = 'pr-license-status pr-error';
      }
    }
  }

  // ─── Plan ───────────────────────────────────────────────────

  /**
   * Plan bilgisini yuklayip render eder.
   */
  async function loadPlanInfo() {
    const response = await sendMessage({ action: 'get_settings' });

    if (response && response.success && response.data) {
      userSettings = Object.assign({}, userSettings, response.data);
    }

    clearChildren(planContent);
    renderPlanComparison(userSettings.plan || 'free');
  }

  /**
   * Plan karsilastirma bolumunu render eder.
   * @param {string} currentPlan
   */
  function renderPlanComparison(currentPlan) {
    const container = el('div', { className: 'pr-fade-in' });

    // Mevcut plan badge
    const planConfig = PLAN_CONFIG[currentPlan] || PLAN_CONFIG.free;
    container.appendChild(el('span', {
      className: 'pr-plan-badge ' + planConfig.cssClass,
      textContent: planConfig.icon + ' ' + planConfig.label
    }));

    // Yukseltme butonu (free plan icin)
    if (currentPlan === 'free') {
      const upgradeBtn = el('button', {
        className: 'pr-upgrade-btn',
        textContent: "\uD83D\uDE80 PRO'ya Yukselt - 149 TL/ay"
      });
      upgradeBtn.addEventListener('click', function () {
        chrome.tabs.create({ url: 'https://pazarradar.com/pro' });
      });
      container.appendChild(upgradeBtn);
    }

    // Ozellik listesi
    const features = [
      'Sinirsiz analiz',
      '3 platform karsilastirma',
      'Satis tahmini',
      'Fiyat gecmisi',
      'Nis skoru'
    ];

    const featureList = el('ul', { className: 'pr-feature-list' });
    for (const feature of features) {
      featureList.appendChild(el('li', { className: 'pr-feature-item' }, [
        el('span', { className: 'pr-feature-check', textContent: '\u2713' }),
        document.createTextNode(feature)
      ]));
    }
    container.appendChild(featureList);

    planContent.appendChild(container);
  }

  // ─── Bos Durum ──────────────────────────────────────────────

  /**
   * Bos durum mesaji gosterir.
   * @param {HTMLElement} container
   * @param {string} icon
   * @param {string} message
   */
  function renderEmptyState(container, icon, message) {
    const emptyEl = el('div', { className: 'pr-empty-state' }, [
      el('span', { className: 'pr-empty-icon', textContent: icon }),
      document.createTextNode(message)
    ]);
    container.appendChild(emptyEl);
  }

  // ─── Platform Tab Yonetimi ──────────────────────────────────

  /**
   * Platform sekmesi degistirme isleyicisi.
   * @param {string} platform
   */
  function switchPlatformTab(platform) {
    activePlatform = platform;

    // Tab stillerini guncelle
    const tabs = document.querySelectorAll('.pr-platform-tab');
    for (const tab of tabs) {
      if (tab.dataset.platform === platform) {
        tab.classList.add('pr-active');
      } else {
        tab.classList.remove('pr-active');
      }
    }

    // Mevcut urun varsa karsilastirma/kar hesaplamayi guncelle
    if (currentProductData) {
      renderProductAnalysis(currentProductData);
    }
  }

  // ─── Mesaj Dinleme (Content Script'lerden) ──────────────────

  /**
   * Gelen mesajlari dinler. Content script'ten urun verisi alinir.
   */
  function setupMessageListener() {
    chrome.runtime.onMessage.addListener(function (msg) {
      if (msg.action === 'current_product_data') {
        renderProductAnalysis(msg.data);
        return;
      }

      if (msg.action === 'price_alert_update') {
        renderPriceAlerts();
        // Bildirim noktasini goster
        const notifDot = document.getElementById('prNotifDot');
        if (notifDot) notifDot.classList.remove('pr-hidden');
        return;
      }

      if (msg.action === 'settings_updated') {
        if (msg.data) {
          userSettings = Object.assign({}, userSettings, msg.data);
        }
        return;
      }
    });
  }

  /**
   * Aktif tab'daki content script'ten urun verisini talep eder.
   */
  async function requestCurrentProductData() {
    try {
      const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
      if (tabs && tabs.length > 0) {
        chrome.tabs.sendMessage(tabs[0].id, { action: 'get_product_data' }, function (response) {
          if (chrome.runtime.lastError) {
            // Content script yuklu degil veya yanit yok - normal durum
            return;
          }
          if (response && response.data) {
            renderProductAnalysis(response.data);
          }
        });
      }
    } catch (_err) {
      // Sessizce basarisiz ol
    }
  }

  // ─── Baslatma ───────────────────────────────────────────────

  /**
   * Panel baslatma fonksiyonu.
   */
  function init() {
    // DOM referanslari
    productSection = document.getElementById('prProductSection');
    productContent = document.getElementById('prProductContent');
    recentContent = document.getElementById('prRecentContent');
    alertsContent = document.getElementById('prAlertsContent');
    settingsSection = document.getElementById('prSettingsSection');
    settingsContent = document.getElementById('prSettingsContent');
    licenseSection = document.getElementById('prLicenseSection');
    licenseContent = document.getElementById('prLicenseContent');
    planContent = document.getElementById('prPlanContent');

    // Platform tab event listener'lari
    const tabButtons = document.querySelectorAll('.pr-platform-tab');
    for (const tab of tabButtons) {
      tab.addEventListener('click', function () {
        switchPlatformTab(tab.dataset.platform);
      });
    }

    // Ayarlar toggle
    const settingsToggle = document.getElementById('prSettingsToggle');
    if (settingsToggle) {
      settingsToggle.addEventListener('click', function () {
        const isHidden = settingsSection.classList.contains('pr-hidden');
        if (isHidden) {
          settingsSection.classList.remove('pr-hidden');
          licenseSection.classList.remove('pr-hidden');
          renderSettings();
          renderLicenseSection();
        } else {
          settingsSection.classList.add('pr-hidden');
          licenseSection.classList.add('pr-hidden');
        }
      });
    }

    // Ayarlari kapat
    const closeSettings = document.getElementById('prCloseSettings');
    if (closeSettings) {
      closeSettings.addEventListener('click', function () {
        settingsSection.classList.add('pr-hidden');
        licenseSection.classList.add('pr-hidden');
      });
    }

    // Bildirim butonu
    const notifBtn = document.getElementById('prNotifBtn');
    if (notifBtn) {
      notifBtn.addEventListener('click', function () {
        // Bildirim noktasini gizle
        const notifDot = document.getElementById('prNotifDot');
        if (notifDot) notifDot.classList.add('pr-hidden');

        // Alarmlar bolumune scroll
        const alertsSection = document.getElementById('prAlertsSection');
        if (alertsSection) {
          alertsSection.scrollIntoView({ behavior: 'smooth' });
        }
      });
    }

    // Mesaj dinleyicisi kur
    setupMessageListener();

    // Asenkron verileri yukle
    loadPlanInfo();
    renderRecentAnalyses();
    renderPriceAlerts();

    // Aktif tab'dan urun verisi talep et
    requestCurrentProductData();
  }

  // DOM hazir olunca baslat
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }
})();
